<?php
/**
 * Utility class for deleting posts.
 *
 * @author     Sudar
 *
 * @package    BulkDelete
 */
defined( 'ABSPATH' ) || exit; // Exit if accessed directly.

class Bulk_Delete_Posts {
	/**
	 * Render post status box.
	 */
	public static function render_delete_posts_by_status_box() {
		if ( BD_Util::is_posts_box_hidden( Bulk_Delete::BOX_POST_STATUS ) ) {
			/* translators: 1 Number of posts that are deleted. */
			printf( __( 'This section just got enabled. Kindly <a href = "%1$s">refresh</a> the page to fully enable it.', 'bulk-delete' ), 'admin.php?page=' . Bulk_Delete::POSTS_PAGE_SLUG );

			return;
		}

		$post_statuses = bd_get_post_statuses();
		$post_count    = wp_count_posts();
?>
		<h4><?php _e( 'Select the post statuses from which you want to delete posts', 'bulk-delete' ); ?></h4>

		<fieldset class="options">
		<table class="optiontable">

			<?php foreach ( $post_statuses as $post_status ) : ?>
				<tr>
					<td>
						<input name="smbd_post_status[]" id="smbd_<?php echo esc_attr( $post_status->name ); ?>"
							value="<?php echo esc_attr( $post_status->name ); ?>" type="checkbox">

						<label for="smbd_<?php echo esc_attr( $post_status->name ); ?>">
							<?php echo esc_html( $post_status->label ), ' '; ?>
							<?php if ( property_exists( $post_count, $post_status->name ) ) : ?>
								(<?php echo absint( $post_count->{ $post_status->name } ) . ' ', __( 'Posts', 'bulk-delete' ); ?>)
							<?php endif; ?>
						</label>
					</td>
				</tr>
			<?php endforeach; ?>

			<?php $sticky_post_count = count( get_option( 'sticky_posts' ) ); ?>

			<tr>
				<td>
					<input name="smbd_sticky" id="smbd_sticky" value="on" type="checkbox">
					<label for="smbd_sticky">
						<?php echo __( 'All Sticky Posts', 'bulk-delete' ), ' '; ?>
						(<?php echo absint( $sticky_post_count ), ' ', __( 'Posts', 'bulk-delete' ); ?>)
						<?php echo '<strong>', __( 'Note', 'bulk-delete' ), '</strong>: ', __( 'The date filter will not work for sticky posts', 'bulk-delete' ); ?>
					</label>
				</td>
			</tr>

		</table>

		<table class="optiontable">
			<?php bd_render_filtering_table_header(); ?>
			<?php bd_render_restrict_settings( 'post_status' ); ?>
			<?php bd_render_delete_settings( 'post_status' ); ?>
			<?php bd_render_limit_settings( 'post_status' ); ?>
			<?php bd_render_cron_settings( 'post_status', 'http://bulkwp.com/addons/scheduler-for-deleting-posts-by-status/?utm_source=wpadmin&utm_campaign=BulkDelete&utm_medium=buynow&utm_content=bd-sps' ); ?>
		</table>

		</fieldset>
<?php
		bd_render_submit_button( 'delete_posts_by_status' );
	}

	/**
	 * Delete posts by post status.
	 *
	 * @since 5.0
	 * @static
	 *
	 * Nonce verification is done in the hook that calls this function.
	 * phpcs:disable WordPress.CSRF.NonceVerification.NoNonceVerification
	 */
	public static function do_delete_posts_by_status() {
		$delete_options = array();

		$delete_options['restrict']     = bd_array_get_bool( $_POST, 'smbd_post_status_restrict', false );
		$delete_options['limit_to']     = absint( bd_array_get( $_POST, 'smbd_post_status_limit_to', 0 ) );
		$delete_options['force_delete'] = bd_array_get_bool( $_POST, 'smbd_post_status_force_delete', false );

		$delete_options['date_op'] = bd_array_get( $_POST, 'smbd_post_status_op' );
		$delete_options['days']    = absint( bd_array_get( $_POST, 'smbd_post_status_days' ) );

		$delete_options['post_status'] = array_map( 'sanitize_text_field', bd_array_get( $_POST, 'smbd_post_status', array() ) );

		$delete_options['delete-sticky-posts'] = bd_array_get_bool( $_POST, 'smbd_sticky', false );

		if ( bd_array_get_bool( $_POST, 'smbd_post_status_cron', false ) ) {
			$freq = sanitize_text_field( $_POST['smbd_post_status_cron_freq'] );
			$time = strtotime( $_POST['smbd_post_status_cron_start'] ) - ( get_option( 'gmt_offset' ) * 60 * 60 );

			if ( -1 === $freq ) {
				wp_schedule_single_event( $time, Bulk_Delete::CRON_HOOK_POST_STATUS, array( $delete_options ) );
			} else {
				wp_schedule_event( $time, $freq, Bulk_Delete::CRON_HOOK_POST_STATUS, array( $delete_options ) );
			}

			$msg = __( 'Posts with the selected status are scheduled for deletion.', 'bulk-delete' ) . ' ';

			/* translators: 1 Url to view cron jobs */
			$msg .= sprintf( __( 'See the full list of <a href = "%s">scheduled tasks</a>', 'bulk-delete' ), get_bloginfo( 'wpurl' ) . '/wp-admin/admin.php?page=' . Bulk_Delete::CRON_PAGE_SLUG );
		} else {
			$deleted_count = self::delete_posts_by_status( $delete_options );

			/* translators: 1 Number of posts deleted */
			$msg = sprintf( _n( 'Deleted %d post with the selected post status', 'Deleted %d posts with the selected post status', $deleted_count, 'bulk-delete' ), $deleted_count );
		}

		add_settings_error(
			Bulk_Delete::POSTS_PAGE_SLUG,
			'deleted-posts',
			$msg,
			'updated'
		);
	} // phpcs:enable

	/**
	 * Delete posts by post status - drafts, pending posts, scheduled posts etc.
	 *
	 * @since  5.0
	 * @static
	 *
	 * @param array $delete_options Options for deleting posts.
	 *
	 * @return int $posts_deleted  Number of posts that were deleted
	 */
	public static function delete_posts_by_status( $delete_options ) {
		$delete_options = bd_convert_old_options_for_delete_post_by_status( $delete_options );
		$delete_options = apply_filters( 'bd_delete_options', $delete_options );

		$posts_deleted = 0;

		if ( $delete_options['delete-sticky-posts'] ) {
			$posts_deleted += self::delete_sticky_posts( $delete_options['force_delete'] );
		}

		if ( empty( $delete_options['post_status'] ) ) {
			return $posts_deleted;
		}

		$options = array(
			'post_status'  => $delete_options['post_status'],
			'post__not_in' => get_option( 'sticky_posts' ),
		);

		$options = bd_build_query_options( $delete_options, $options );

		$post_ids = bd_query( $options );
		foreach ( $post_ids as $post_id ) {
			wp_delete_post( $post_id, $delete_options['force_delete'] );
		}

		$posts_deleted += count( $post_ids );

		return $posts_deleted;
	}

	/**
	 * Delete all sticky posts.
	 *
	 * @since 5.6.0
	 *
	 * @param bool $force_delete Whether to force delete the posts.
	 *
	 * @return int Number of posts deleted.
	 */
	public static function delete_sticky_posts( $force_delete ) {
		$sticky_post_ids = get_option( 'sticky_posts' );

		foreach ( $sticky_post_ids as $sticky_post_id ) {
			wp_delete_post( $sticky_post_id, $force_delete );
		}

		return count( $sticky_post_ids );
	}

	/**
	 * Render Delete posts by category box.
	 */
	public static function render_delete_posts_by_category_box() {
		if ( BD_Util::is_posts_box_hidden( Bulk_Delete::BOX_CATEGORY ) ) {
			printf( __( 'This section just got enabled. Kindly <a href = "%1$s">refresh</a> the page to fully enable it.', 'bulk-delete' ), 'admin.php?page=' . Bulk_Delete::POSTS_PAGE_SLUG );

			return;
		}
?>
        <!-- Category Start-->
        <h4><?php _e( 'Select the post type from which you want to delete posts by category', 'bulk-delete' ); ?></h4>
        <fieldset class="options">
        <table class="optiontable">
			<?php bd_render_post_type_dropdown( 'cats' ); ?>
        </table>

        <h4><?php _e( 'Select the categories from which you wan to delete posts', 'bulk-delete' ); ?></h4>
        <p><?php _e( 'Note: The post count below for each category is the total number of posts in that category, irrespective of post type', 'bulk-delete' ); ?>.</p>
<?php
		$categories = get_categories( array(
				'hide_empty' => false,
			)
		);
?>
        <table class="form-table">
            <tr>
                <td scope="row">
				<select class="select2" name="smbd_cats[]" multiple data-placeholder="<?php _e( 'Select Categories', 'bulk-delete' ); ?>">
					<option value="all"><?php _e( 'All Categories', 'bulk-delete' ); ?></option>
		<?php foreach ( $categories as $category ) { ?>
			<option value="<?php echo $category->cat_ID; ?>"><?php echo $category->cat_name, ' (', $category->count, ' ', __( 'Posts', 'bulk-delete' ), ')'; ?></option>
		<?php } ?>
					</select>
                </td>
            </tr>
        </table>

        <table class="optiontable">
			<?php bd_render_filtering_table_header(); ?>
			<?php bd_render_restrict_settings( 'cats' ); ?>
			<?php bd_render_delete_settings( 'cats' ); ?>
			<?php bd_render_private_post_settings( 'cats' ); ?>
			<?php bd_render_limit_settings( 'cats' ); ?>
			<?php bd_render_cron_settings( 'cats', 'http://bulkwp.com/addons/scheduler-for-deleting-posts-by-category/?utm_source=wpadmin&utm_campaign=BulkDelete&utm_medium=buynow&utm_content=bd-sc' ); ?>
        </table>

        </fieldset>
<?php
		bd_render_submit_button( 'delete_posts_by_category' );
	}

	/**
	 * Process delete posts by category.
	 *
	 * @since 5.0
	 * @static
	 */
	public static function do_delete_posts_by_category() {
		$delete_options = array();

		$delete_options['post_type']     = array_get( $_POST, 'smbd_cats_post_type', 'post' );
		$delete_options['selected_cats'] = array_get( $_POST, 'smbd_cats' );
		$delete_options['restrict']      = array_get_bool( $_POST, 'smbd_cats_restrict', false );
		$delete_options['private']       = array_get_bool( $_POST, 'smbd_cats_private', false );
		$delete_options['limit_to']      = absint( array_get( $_POST, 'smbd_cats_limit_to', 0 ) );
		$delete_options['force_delete']  = array_get_bool( $_POST, 'smbd_cats_force_delete', false );

		$delete_options['date_op']       = array_get( $_POST, 'smbd_cats_op' );
		$delete_options['days']          = absint( array_get( $_POST, 'smbd_cats_days' ) );

		if ( array_get_bool( $_POST, 'smbd_cats_cron', false ) ) {
			$freq = $_POST['smbd_cats_cron_freq'];
			$time = strtotime( $_POST['smbd_cats_cron_start'] ) - ( get_option( 'gmt_offset' ) * 60 * 60 );

			if ( $freq == -1 ) {
				wp_schedule_single_event( $time, Bulk_Delete::CRON_HOOK_CATEGORY, array( $delete_options ) );
			} else {
				wp_schedule_event( $time, $freq , Bulk_Delete::CRON_HOOK_CATEGORY, array( $delete_options ) );
			}

			$msg = __( 'Posts from the selected categories are scheduled for deletion.', 'bulk-delete' ) . ' ' .
				sprintf( __( 'See the full list of <a href = "%s">scheduled tasks</a>' , 'bulk-delete' ), get_bloginfo( 'wpurl' ) . '/wp-admin/admin.php?page=' . Bulk_Delete::CRON_PAGE_SLUG );
		} else {
			$deleted_count = self::delete_posts_by_category( $delete_options );
			$msg           = sprintf( _n( 'Deleted %d post from the selected categories', 'Deleted %d posts from the selected categories' , $deleted_count, 'bulk-delete' ), $deleted_count );
		}

		add_settings_error(
			Bulk_Delete::POSTS_PAGE_SLUG,
			'deleted-posts',
			$msg,
			'updated'
		);
	}

	/**
	 * Delete posts by category.
	 *
	 * @since 5.0
	 * @static
	 *
	 * @param array $delete_options Options for deleting posts
	 *
	 * @return int $posts_deleted  Number of posts that were deleted
	 */
	public static function delete_posts_by_category( $delete_options ) {
		// Backward compatibility code. Will be removed in Bulk Delete v6.0
		$delete_options['post_type'] = array_get( $delete_options, 'post_type', 'post' );

		if ( array_key_exists( 'cats_op', $delete_options ) ) {
			$delete_options['date_op'] = $delete_options['cats_op'];
			$delete_options['days']    = $delete_options['cats_days'];
		}

		$delete_options = apply_filters( 'bd_delete_options', $delete_options );

		$options       = array();
		$selected_cats = $delete_options['selected_cats'];
		if ( in_array( 'all', $selected_cats ) ) {
			$options['category__not__in'] = array(0);
		} else {
			$options['category__in'] = $selected_cats;
		}

		$options  = bd_build_query_options( $delete_options, $options );
		$post_ids = bd_query( $options );

		foreach ( $post_ids as $post_id ) {
			// $force delete parameter to custom post types doesn't work
			if ( $delete_options['force_delete'] ) {
				wp_delete_post( $post_id, true );
			} else {
				wp_trash_post( $post_id );
			}
		}

		return count( $post_ids );
	}

	/**
	 * Render delete posts by tag box.
	 */
	public static function render_delete_posts_by_tag_box() {
		if ( BD_Util::is_posts_box_hidden( Bulk_Delete::BOX_TAG ) ) {
			printf( __( 'This section just got enabled. Kindly <a href = "%1$s">refresh</a> the page to fully enable it.', 'bulk-delete' ), 'admin.php?page=' . Bulk_Delete::POSTS_PAGE_SLUG );

			return;
		}

		$tags = get_tags();
		if ( count( $tags ) > 0 ) {
?>
            <h4><?php _e( 'Select the tags from which you want to delete posts', 'bulk-delete' ) ?></h4>

            <!-- Tags start-->
            <fieldset class="options">
            <table class="form-table">
                <tr>
				<td scope="row" colspan="2">
					<select class="select2" name="smbd_tags[]" multiple data-placeholder="<?php _e( 'Select Tags', 'bulk-delete' ); ?>">
					<option value="all"><?php _e( 'All Tags', 'bulk-delete' ); ?></option>
				<?php foreach ( $tags as $tag ) { ?>
					<option value="<?php echo absint( $tag->term_id ); ?>"><?php echo $tag->name, ' (', $tag->count, ' ', __( 'Posts', 'bulk-delete' ), ')'; ?></option>
				<?php } ?>
					</select>
				</td>
                </tr>
			</table>

            <table class="optiontable">
				<?php bd_render_filtering_table_header(); ?>
				<?php bd_render_restrict_settings( 'tags' ); ?>
				<?php bd_render_delete_settings( 'tags' ); ?>
				<?php bd_render_private_post_settings( 'tags' ); ?>
				<?php bd_render_limit_settings( 'tags' ); ?>
				<?php bd_render_cron_settings( 'tags', 'http://bulkwp.com/addons/scheduler-for-deleting-posts-by-tag/?utm_source=wpadmin&utm_campaign=BulkDelete&utm_medium=buynow&utm_content=bd-st' ); ?>
            </table>
            </fieldset>
<?php
			bd_render_submit_button( 'delete_posts_by_tag' );
		} else {
?>
            <h4><?php _e( "You don't have any posts assigned to tags in this blog.", 'bulk-delete' ) ?></h4>
<?php
		}
	}

	/**
	 * Process Delete Posts by tag request.
	 *
	 * @static
	 *
	 * @since 5.0
	 */
	public static function do_delete_posts_by_tag() {
		$delete_options                  = array();
		$delete_options['selected_tags'] = array_get( $_POST, 'smbd_tags' );
		$delete_options['restrict']      = array_get_bool( $_POST, 'smbd_tags_restrict', false );
		$delete_options['private']       = array_get( $_POST, 'smbd_tags_private' );
		$delete_options['limit_to']      = absint( array_get( $_POST, 'smbd_tags_limit_to', 0 ) );
		$delete_options['force_delete']  = array_get_bool( $_POST, 'smbd_tags_force_delete', false );

		$delete_options['date_op']       = array_get( $_POST, 'smbd_tags_op' );
		$delete_options['days']          = absint( array_get( $_POST, 'smbd_tags_days' ) );

		if ( array_get( $_POST, 'smbd_tags_cron', 'false' ) == 'true' ) {
			$freq = $_POST['smbd_tags_cron_freq'];
			$time = strtotime( $_POST['smbd_tags_cron_start'] ) - ( get_option( 'gmt_offset' ) * 60 * 60 );

			if ( $freq == -1 ) {
				wp_schedule_single_event( $time, Bulk_Delete::CRON_HOOK_TAG, array( $delete_options ) );
			} else {
				wp_schedule_event( $time, $freq, Bulk_Delete::CRON_HOOK_TAG, array( $delete_options ) );
			}
			$msg = __( 'Posts from the selected tags are scheduled for deletion.', 'bulk-delete' ) . ' ' .
				sprintf( __( 'See the full list of <a href = "%s">scheduled tasks</a>' , 'bulk-delete' ), get_bloginfo( 'wpurl' ) . '/wp-admin/admin.php?page=' . Bulk_Delete::CRON_PAGE_SLUG );
		} else {
			$deleted_count = self::delete_posts_by_tag( $delete_options );
			$msg           = sprintf( _n( 'Deleted %d post from the selected tags', 'Deleted %d posts from the selected tags' , $deleted_count, 'bulk-delete' ), $deleted_count );
		}

		add_settings_error(
			Bulk_Delete::POSTS_PAGE_SLUG,
			'deleted-posts',
			$msg,
			'updated'
		);
	}

	/**
	 * Delete posts by tag.
	 *
	 * @since 5.0
	 * @static
	 *
	 * @param array $delete_options Options for deleting posts
	 *
	 * @return int $posts_deleted  Number of posts that were deleted
	 */
	public static function delete_posts_by_tag( $delete_options ) {
		// Backward compatibility code. Will be removed in Bulk Delete v6.0
		if ( array_key_exists( 'tags_op', $delete_options ) ) {
			$delete_options['date_op'] = $delete_options['tags_op'];
			$delete_options['days']    = $delete_options['tags_days'];
		}

		$delete_options = apply_filters( 'bd_delete_options', $delete_options );

		$options       = array();
		$selected_tags = $delete_options['selected_tags'];
		if ( in_array( 'all', $selected_tags ) ) {
			$options['tag__not__in'] = array(0);
		} else {
			$options['tag__in'] = $selected_tags;
		}

		$options  = bd_build_query_options( $delete_options, $options );
		$post_ids = bd_query( $options );
		foreach ( $post_ids as $post_id ) {
			wp_delete_post( $post_id, $delete_options['force_delete'] );
		}

		return count( $post_ids );
	}

	/**
	 * Render delete by custom taxonomy box.
	 */
	public static function render_delete_posts_by_taxonomy_box() {
		if ( BD_Util::is_posts_box_hidden( Bulk_Delete::BOX_TAX ) ) {
			printf( __( 'This section just got enabled. Kindly <a href = "%1$s">refresh</a> the page to fully enable it.', 'bulk-delete' ), 'admin.php?page=' . Bulk_Delete::POSTS_PAGE_SLUG );

			return;
		}

		$taxs =  get_taxonomies( array(
				'public'   => true,
				'_builtin' => false,
			), 'objects'
		);

		$terms_array = array();
		if ( count( $taxs ) > 0 ) {
			foreach ( $taxs as $tax ) {
				$terms = get_terms( $tax->name );
				if ( count( $terms ) > 0 ) {
					$terms_array[$tax->name] = $terms;
				}
			}
		}

		if ( count( $terms_array ) > 0 ) {
?>
        <!-- Custom tax Start-->
        <h4><?php _e( 'Select the post type from which you want to delete posts by custom taxonomy', 'bulk-delete' ); ?></h4>

        <fieldset class="options">
            <table class="optiontable">
				<?php bd_render_post_type_dropdown( 'tax' ); ?>
            </table>

            <h4><?php _e( 'Select the taxonomies from which you want to delete posts', 'bulk-delete' ) ?></h4>

            <table class="optiontable">
<?php
			foreach ( $terms_array as $tax => $terms ) {
?>
                <tr>
                    <td scope="row" >
                        <input name="smbd_taxs" value="<?php echo $tax; ?>" type="radio" class="custom-tax">
                    </td>
                    <td>
                        <label for="smbd_taxs"><?php echo $taxs[$tax]->labels->name; ?> </label>
                    </td>
                </tr>
<?php
			}
?>
            </table>

            <h4><?php _e( 'The selected taxonomy has the following terms. Select the terms from which you want to delete posts', 'bulk-delete' ) ?></h4>
            <p><?php _e( 'Note: The post count below for each term is the total number of posts in that term, irrespective of post type', 'bulk-delete' ); ?>.</p>
<?php
			foreach ( $terms_array as $tax => $terms ) {
?>
            <table class="optiontable terms_<?php echo $tax;?> terms">
<?php
				foreach ( $terms as $term ) {
?>
                    <tr>
                        <td scope="row" >
                            <input name="smbd_tax_terms[]" value="<?php echo $term->slug; ?>" type="checkbox" class="terms">
                        </td>
                        <td>
                            <label for="smbd_tax_terms"><?php echo $term->name; ?> (<?php echo $term->count . ' '; _e( 'Posts', 'bulk-delete' ); ?>)</label>
                        </td>
                    </tr>
<?php
				}
?>
            </table>
<?php
			}
?>
            <table class="optiontable">
				<?php bd_render_filtering_table_header(); ?>
				<?php bd_render_restrict_settings( 'taxs' ); ?>
				<?php bd_render_delete_settings( 'taxs' ); ?>
				<?php bd_render_private_post_settings( 'taxs' ); ?>
				<?php bd_render_limit_settings( 'taxs' ); ?>
				<?php bd_render_cron_settings( 'taxs', 'http://bulkwp.com/addons/scheduler-for-deleting-posts-by-taxonomy/?utm_source=wpadmin&utm_campaign=BulkDelete&utm_medium=buynow&utm_content=bd-stx' ); ?>
            </table>

            </fieldset>
<?php
			bd_render_submit_button( 'delete_posts_by_taxonomy' );
		} else {
?>
            <h4><?php _e( "This WordPress installation doesn't have any non-empty custom taxonomies defined", 'bulk-delete' ) ?></h4>
<?php
		}
	}

	/**
	 * Process Delete posts by Taxonomy Request.
	 *
	 * @static
	 *
	 * @since 5.0
	 */
	public static function do_delete_posts_by_taxonomy() {
		$delete_options                       = array();
		$delete_options['post_type']          = array_get( $_POST, 'smbd_tax_post_type', 'post' );
		$delete_options['selected_taxs']      = array_get( $_POST, 'smbd_taxs' );
		$delete_options['selected_tax_terms'] = array_get( $_POST, 'smbd_tax_terms' );
		$delete_options['restrict']           = array_get_bool( $_POST, 'smbd_taxs_restrict', false );
		$delete_options['private']            = array_get_bool( $_POST, 'smbd_taxs_private' );
		$delete_options['limit_to']           = absint( array_get( $_POST, 'smbd_taxs_limit_to', 0 ) );
		$delete_options['force_delete']       = array_get_bool( $_POST, 'smbd_taxs_force_delete', false );

		$delete_options['date_op']            = array_get( $_POST, 'smbd_taxs_op' );
		$delete_options['days']               = absint( array_get( $_POST, 'smbd_taxs_days' ) );

		if ( array_get( $_POST, 'smbd_taxs_cron', 'false' ) == 'true' ) {
			$freq = $_POST['smbd_taxs_cron_freq'];
			$time = strtotime( $_POST['smbd_taxs_cron_start'] ) - ( get_option( 'gmt_offset' ) * 60 * 60 );

			if ( $freq == -1 ) {
				wp_schedule_single_event( $time, Bulk_Delete::CRON_HOOK_TAXONOMY, array( $delete_options ) );
			} else {
				wp_schedule_event( $time, $freq, Bulk_Delete::CRON_HOOK_TAXONOMY, array( $delete_options ) );
			}
			$msg = __( 'Posts from the selected custom taxonomies are scheduled for deletion.', 'bulk-delete' ) . ' ' .
				sprintf( __( 'See the full list of <a href = "%s">scheduled tasks</a>' , 'bulk-delete' ), get_bloginfo( 'wpurl' ) . '/wp-admin/admin.php?page=' . Bulk_Delete::CRON_PAGE_SLUG );
		} else {
			$deleted_count = self::delete_posts_by_taxonomy( $delete_options );
			$msg           = sprintf( _n( 'Deleted %d post from the selected custom taxonomies', 'Deleted %d posts from the selected custom taxonomies' , $deleted_count, 'bulk-delete' ), $deleted_count );
		}

		add_settings_error(
			Bulk_Delete::POSTS_PAGE_SLUG,
			'deleted-posts',
			$msg,
			'updated'
		);
	}

	/**
	 * Delete posts by custom taxonomy.
	 *
	 * @since 5.0
	 * @static
	 *
	 * @param array $delete_options Options for deleting posts
	 *
	 * @return int $posts_deleted  Number of posts that were deleted
	 */
	public static function delete_posts_by_taxonomy( $delete_options ) {
		// For compatibility reasons set default post type to 'post'
		$post_type = array_get( $delete_options, 'post_type', 'post' );

		if ( array_key_exists( 'taxs_op', $delete_options ) ) {
			$delete_options['date_op'] = $delete_options['taxs_op'];
			$delete_options['days']    = $delete_options['taxs_days'];
		}

		$delete_options = apply_filters( 'bd_delete_options', $delete_options );

		$selected_taxs      = $delete_options['selected_taxs'];
		$selected_tax_terms = $delete_options['selected_tax_terms'];

		$options = array(
			'post_status' => 'publish',
			'post_type'   => $post_type,
			'tax_query'   => array(
				array(
					'taxonomy' => $selected_taxs,
					'terms'    => $selected_tax_terms,
					'field'    => 'slug',
				),
			),
		);

		$options  = bd_build_query_options( $delete_options, $options );
		$post_ids = bd_query( $options );
		foreach ( $post_ids as $post_id ) {
			// $force delete parameter to custom post types doesn't work
			if ( $delete_options['force_delete'] ) {
				wp_delete_post( $post_id, true );
			} else {
				wp_trash_post( $post_id );
			}
		}

		return count( $post_ids );
	}

	/**
	 * Render delete by custom post type box.
	 *
	 * @static
	 */
	public static function render_delete_posts_by_post_type_box() {
		if ( BD_Util::is_posts_box_hidden( Bulk_Delete::BOX_POST_TYPE ) ) {
			printf( __( 'This section just got enabled. Kindly <a href = "%1$s">refresh</a> the page to fully enable it.', 'bulk-delete' ), 'admin.php?page=' . Bulk_Delete::POSTS_PAGE_SLUG );

			return;
		}

		$types_array = array();

		$types = get_post_types( array(
				'_builtin' => false,
			), 'names'
		);

		if ( count( $types ) > 0 ) {
			foreach ( $types as $type ) {
				$post_count = wp_count_posts( $type );
				if ( $post_count->publish > 0 ) {
					$types_array["$type-publish"] = $post_count->publish;
				}
				if ( $post_count->future > 0 ) {
					$types_array["$type-future"] = $post_count->future;
				}
				if ( $post_count->pending > 0 ) {
					$types_array["$type-pending"] = $post_count->pending;
				}
				if ( $post_count->draft > 0 ) {
					$types_array["$type-draft"] = $post_count->draft;
				}
				if ( $post_count->private > 0 ) {
					$types_array["$type-private"] = $post_count->private;
				}
			}
		}

		if ( count( $types_array ) > 0 ) {
?>
            <!-- Custom post type Start-->
            <h4><?php _e( 'Select the custom post types from which you want to delete posts', 'bulk-delete' ) ?></h4>

            <fieldset class="options">
            <table class="optiontable">
<?php
			foreach ( $types_array as $type => $count ) {
?>
                <tr>
                    <td scope="row" >
                        <input name="smbd_types[]" value="<?php echo $type; ?>" type="checkbox">
                    </td>
                    <td>
						<label for="smbd_types"><?php echo BD_Util::display_post_type_status( $type ), ' (', $count, ')'; ?></label>
                    </td>
                </tr>
<?php
			}

			bd_render_filtering_table_header();
			bd_render_restrict_settings( 'types' );
			bd_render_delete_settings( 'types' );
			bd_render_limit_settings( 'types' );
			bd_render_cron_settings( 'types', 'http://bulkwp.com/addons/scheduler-for-deleting-posts-by-post-type/?utm_source=wpadmin&utm_campaign=BulkDelete&utm_medium=buynow&utm_content=bd-spt' );
?>
            </table>
            </fieldset>
<?php
			bd_render_submit_button( 'delete_posts_by_post_type' );
		} else {
            printf( '<h4>%s</h4>', __( "This WordPress installation doesn't have any non-empty custom post types", 'bulk-delete' ) );
		}
	}

	/**
	 * Process request to delete posts by post type.
	 *
	 * @static
	 *
	 * @since 5.0
	 */
	public static function do_delete_posts_by_post_type() {
		$delete_options                   = array();

		$delete_options['selected_types'] = array_get( $_POST, 'smbd_types' );
		$delete_options['restrict']       = array_get_bool( $_POST, 'smbd_types_restrict', false );
		$delete_options['limit_to']       = absint( array_get( $_POST, 'smbd_types_limit_to', 0 ) );
		$delete_options['force_delete']   = array_get_bool( $_POST, 'smbd_types_force_delete', false );

		$delete_options['date_op']        = array_get( $_POST, 'smbd_types_op' );
		$delete_options['days']           = absint( array_get( $_POST, 'smbd_types_days' ) );

		if ( array_get( $_POST, 'smbd_types_cron', 'false' ) == 'true' ) {
			$freq = $_POST['smbd_types_cron_freq'];
			$time = strtotime( $_POST['smbd_types_cron_start'] ) - ( get_option( 'gmt_offset' ) * 60 * 60 );

			if ( $freq == -1 ) {
				wp_schedule_single_event( $time, Bulk_Delete::CRON_HOOK_POST_TYPE, array( $delete_options ) );
			} else {
				wp_schedule_event( $time, $freq, Bulk_Delete::CRON_HOOK_POST_TYPE, array( $delete_options ) );
			}

			$msg = __( 'Posts from the selected custom post type are scheduled for deletion.', 'bulk-delete' ) . ' ' .
				sprintf( __( 'See the full list of <a href = "%s">scheduled tasks</a>' , 'bulk-delete' ), get_bloginfo( 'wpurl' ) . '/wp-admin/admin.php?page=' . Bulk_Delete::CRON_PAGE_SLUG );
		} else {
			$deleted_count = self::delete_posts_by_post_type( $delete_options );
			$msg           = sprintf( _n( 'Deleted %d post from the selected custom post type', 'Deleted %d posts from the selected custom post type' , $deleted_count, 'bulk-delete' ), $deleted_count );
		}

		add_settings_error(
			Bulk_Delete::POSTS_PAGE_SLUG,
			'deleted-posts',
			$msg,
			'updated'
		);
	}

	/**
	 * Delete posts by custom post type.
	 *
	 * @static
	 *
	 * @since  5.0
	 *
	 * @param array $delete_options Options for deleting posts
	 *
	 * @return int $posts_deleted  Number of posts that were deleted
	 */
	public static function delete_posts_by_post_type( $delete_options ) {
		// Backward compatibility code. Will be removed in Bulk Delete v6.0
		if ( array_key_exists( 'types_op', $delete_options ) ) {
			$delete_options['date_op'] = $delete_options['types_op'];
			$delete_options['days']    = $delete_options['types_days'];
		}

		$delete_options = apply_filters( 'bd_delete_options', $delete_options );

		$count          = 0;
		$selected_types = $delete_options['selected_types'];

		foreach ( $selected_types as $selected_type ) {
			$type_status = BD_Util::split_post_type_status( $selected_type );

			$type        = $type_status['type'];
			$status      = $type_status['status'];

			$options = array(
				'post_status' => $status,
				'post_type'   => $type,
			);

			$options  = bd_build_query_options( $delete_options, $options );
			$post_ids = bd_query( $options );
			foreach ( $post_ids as $post_id ) {
				// $force delete parameter to custom post types doesn't work
				if ( $delete_options['force_delete'] ) {
					wp_delete_post( $post_id, true );
				} else {
					wp_trash_post( $post_id );
				}
			}

			$count += count( $post_ids );
		}

		return $count;
	}

	/**
	 * Render delete by url box.
	 *
	 * @static
	 */
	public static function render_delete_posts_by_url_box() {
		if ( BD_Util::is_posts_box_hidden( Bulk_Delete::BOX_URL ) ) {
			printf( __( 'This section just got enabled. Kindly <a href = "%1$s">refresh</a> the page to fully enable it.', 'bulk-delete' ), 'admin.php?page=' . Bulk_Delete::POSTS_PAGE_SLUG );

			return;
		}
?>
        <!-- URLs start-->
        <h4><?php _e( 'Delete posts and pages that have the following Permalink', 'bulk-delete' ); ?></h4>

        <fieldset class="options">
        <table class="optiontable">
            <tr>
                <td scope="row" colspan="2">
                    <label for="smdb_specific_pages"><?php _e( 'Enter one post url (not post ids) per line', 'bulk-delete' ); ?></label>
                    <br>
                    <textarea style="width: 450px; height: 80px;" id="smdb_specific_pages_urls" name="smdb_specific_pages_urls" rows="5" columns="80"></textarea>
                </td>
            </tr>

			<?php bd_render_filtering_table_header(); ?>
			<?php bd_render_delete_settings( 'specific' ); ?>

        </table>
        </fieldset>
<?php
		bd_render_submit_button( 'delete_posts_by_url' );
	}

	/**
	 * Delete posts by url.
	 *
	 * @static
	 *
	 * @since 5.0
	 */
	public static function do_delete_posts_by_url() {
		$force_delete = array_get_bool( $_POST, 'smbd_specific_force_delete', false );

		$urls = preg_split( '/\r\n|\r|\n/', array_get( $_POST, 'smdb_specific_pages_urls' ) );
		foreach ( $urls as $url ) {
			$checkedurl = $url;
			if ( substr( $checkedurl , 0, 1 ) == '/' ) {
				$checkedurl = get_site_url() . $checkedurl ;
			}
			$postid = url_to_postid( $checkedurl );
			wp_delete_post( $postid, $force_delete );
		}

		$deleted_count = count( $urls );
		$msg           = sprintf( _n( 'Deleted %d post with the specified urls', 'Deleted %d posts with the specified urls' , $deleted_count, 'bulk-delete' ), $deleted_count );

		add_settings_error(
			Bulk_Delete::POSTS_PAGE_SLUG,
			'deleted-posts',
			$msg,
			'updated'
		);
	}

	/**
	 * Render delete by post revisions box.
	 *
	 * @static
	 */
	public static function render_posts_by_revision_box() {
		global $wpdb;

		if ( BD_Util::is_posts_box_hidden( Bulk_Delete::BOX_POST_REVISION ) ) {
			printf( __( 'This section just got enabled. Kindly <a href = "%1$s">refresh</a> the page to fully enable it.', 'bulk-delete' ), 'admin.php?page=' . Bulk_Delete::POSTS_PAGE_SLUG );

			return;
		}

		$revisions = $wpdb->get_var( "select count(*) from $wpdb->posts where post_type = 'revision'" );
?>
        <!-- Post Revisions start-->
        <h4><?php _e( 'Select the posts which you want to delete', 'bulk-delete' ); ?></h4>

        <fieldset class="options">
        <table class="optiontable">
            <tr>
                <td>
                    <input name="smbd_revisions" id ="smbd_revisions" value="revisions" type="checkbox">
                    <label for="smbd_revisions"><?php _e( 'All Revisions', 'bulk-delete' ); ?> (<?php echo $revisions . ' '; _e( 'Revisions', 'bulk-delete' ); ?>)</label>
                </td>
            </tr>

        </table>
        </fieldset>
<?php
		bd_render_submit_button( 'delete_posts_by_revision' );
	}

	/**
	 * Process delete revisions request.
	 *
	 * @static
	 *
	 * @since 5.0
	 */
	public static function do_delete_posts_by_revision() {
		$delete_options = array( 'revisions' => array_get( $_POST, 'smbd_revisions' ) );

		$deleted_count = self::delete_posts_by_revision( $delete_options );

		$msg = sprintf( _n( 'Deleted %d post revision', 'Deleted %d post revisions' , $deleted_count, 'bulk-delete' ), $deleted_count );

		add_settings_error(
			Bulk_Delete::POSTS_PAGE_SLUG,
			'deleted-posts',
			$msg,
			'updated'
		);
	}

	/**
	 * Delete all post revisions.
	 *
	 * @since 5.0
	 * @static
	 *
	 * @param array $delete_options
	 *
	 * @return int The number of posts that were deleted
	 */
	public static function delete_posts_by_revision( $delete_options ) {
		global $wpdb;

		// Revisions
		if ( 'revisions' == $delete_options['revisions'] ) {
			$revisions = $wpdb->get_results( "select ID from $wpdb->posts where post_type = 'revision'" );

			foreach ( $revisions as $revision ) {
				wp_delete_post( $revision->ID );
			}

			return count( $revisions );
		}

		return 0;
	}

	/**
	 * Filter JS Array and add validation hooks.
	 *
	 * @since 5.4
	 * @static
	 *
	 * @param array $js_array JavaScript Array
	 *
	 * @return array Modified JavaScript Array
	 */
	public static function filter_js_array( $js_array ) {
		$js_array['msg']['deletePostsWarning'] = __( 'Are you sure you want to delete all the posts based on the selected option?', 'bulk-delete' );
		$js_array['msg']['selectPostOption']   = __( 'Please select posts from at least one option', 'bulk-delete' );

		$js_array['validators']['delete_posts_by_category'] = 'validateSelect2';
		$js_array['error_msg']['delete_posts_by_category']  = 'selectCategory';
		$js_array['msg']['selectCategory']                  = __( 'Please select at least one category', 'bulk-delete' );

		$js_array['validators']['delete_posts_by_tag']     = 'validateSelect2';
		$js_array['error_msg']['delete_posts_by_category'] = 'selectTag';
		$js_array['msg']['selectTag']                      = __( 'Please select at least one tag', 'bulk-delete' );

		$js_array['validators']['delete_posts_by_url'] = 'validateUrl';
		$js_array['error_msg']['delete_posts_by_url']  = 'enterUrl';
		$js_array['msg']['enterUrl']                   = __( 'Please enter at least one post url', 'bulk-delete' );

		$js_array['dt_iterators'][] = '_cats';
		$js_array['dt_iterators'][] = '_tags';
		$js_array['dt_iterators'][] = '_taxs';
		$js_array['dt_iterators'][] = '_types';
		$js_array['dt_iterators'][] = '_post_status';

		return $js_array;
	}

	/**
	 * Process delete cron job request.
	 * This should ideally go in a separate class. But I was
	 * lazy to create a separate class for a single function.
	 *
	 * @since 5.0
	 * @static
	 */
	public static function do_delete_cron() {
		$cron_id    = absint( $_GET['cron_id'] );
		$cron_items = BD_Util::get_cron_schedules();
		wp_unschedule_event( $cron_items[$cron_id]['timestamp'], $cron_items[$cron_id]['type'], $cron_items[$cron_id]['args'] );

		$msg = __( 'The selected scheduled job was successfully deleted ', 'bulk-delete' );

		add_settings_error(
			Bulk_Delete::CRON_PAGE_SLUG,
			'deleted-cron',
			$msg,
			'updated'
		);
	}
}

// hooks
add_action( 'bd_delete_posts_by_status'    , array( 'Bulk_Delete_Posts', 'do_delete_posts_by_status' ) );
add_action( 'bd_delete_posts_by_category'  , array( 'Bulk_Delete_Posts', 'do_delete_posts_by_category' ) );
add_action( 'bd_delete_posts_by_tag'       , array( 'Bulk_Delete_Posts', 'do_delete_posts_by_tag' ) );
add_action( 'bd_delete_posts_by_taxonomy'  , array( 'Bulk_Delete_Posts', 'do_delete_posts_by_taxonomy' ) );
add_action( 'bd_delete_posts_by_post_type' , array( 'Bulk_Delete_Posts', 'do_delete_posts_by_post_type' ) );
add_action( 'bd_delete_posts_by_url'       , array( 'Bulk_Delete_Posts', 'do_delete_posts_by_url' ) );
add_action( 'bd_delete_posts_by_revision'  , array( 'Bulk_Delete_Posts', 'do_delete_posts_by_revision' ) );

add_action( 'bd_delete_cron'               , array( 'Bulk_Delete_Posts', 'do_delete_cron' ) );
add_filter( 'bd_javascript_array'          , array( 'Bulk_Delete_Posts', 'filter_js_array' ) );
?>
