<?php
/**
 * Plugin Name: STP Transporte Pro
 * Description: Venda de passagens, cadastro de rotas com desconto padrão e horários, relatórios e filtros por usuário.
 * Version: 1.1.0
 * Author: Hemerson & ChatGPT
 * License: GPLv2 or later
 */

if ( ! defined( 'ABSPATH' ) ) { exit; }

define('STPTP_VERSION', '1.1.0');
define('STPTP_SLUG', 'stptp');
define('STPTP_TABLE_ROTAS', $GLOBALS['wpdb']->prefix . 'stptp_rotas');

/**
 * Ativação: cria tabelas
 */
function stptp_activate(){
    global $wpdb;
    require_once ABSPATH . 'wp-admin/includes/upgrade.php';
    $charset_collate = $wpdb->get_charset_collate();
    $table = STPTP_TABLE_ROTAS;

    $sql = "CREATE TABLE {$table} (
        id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
        origem VARCHAR(100) NOT NULL,
        destino VARCHAR(100) NOT NULL,
        valor_passagem DECIMAL(10,2) NOT NULL DEFAULT 0.00,
        tarifa DECIMAL(10,2) NOT NULL DEFAULT 0.00,
        desconto_padrao DECIMAL(5,2) NOT NULL DEFAULT 0.00, /* Percentual 0-100 */
        hora_partida TIME NULL,
        hora_chegada TIME NULL,
        ativo TINYINT(1) NOT NULL DEFAULT 1,
        created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        KEY idx_origem (origem),
        KEY idx_destino (destino)
    ) {$charset_collate};";

    dbDelta($sql);
}
register_activation_hook(__FILE__, 'stptp_activate');

/**
 * Admin Menu
 */
function stptp_register_menu(){
    add_menu_page(
        __('Transporte', 'stptp'),
        __('Transporte', 'stptp'),
        'edit_posts',
        'stptp-dashboard',
        'stptp_render_dashboard',
        'dashicons-tickets-alt',
        30
    );

    add_submenu_page('stptp-dashboard',
        __('Rotas', 'stptp'), __('Rotas', 'stptp'),
        'edit_posts', 'stptp-rotas', 'stptp_render_rotas_page'
    );

    // Espaço para futuras telas (Vendas, Relatórios)
    // add_submenu_page('stptp-dashboard', 'Vendas', 'Vendas', 'edit_posts', 'stptp-vendas', 'stptp_render_vendas_page');
    // add_submenu_page('stptp-dashboard', 'Relatórios', 'Relatórios', 'edit_posts', 'stptp-relatorios', 'stptp_render_relatorios_page');
}
add_action('admin_menu', 'stptp_register_menu');

/**
 * Enqueue Bootstrap apenas nas telas do plugin
 */
function stptp_admin_assets($hook){
    $screen = isset($_GET['page']) ? sanitize_text_field($_GET['page']) : '';
    $targets = array('stptp-dashboard', 'stptp-rotas');
    if (in_array($screen, $targets, true)){
        // Bootstrap 5 (CSS + Bundle JS) via CDN
        wp_enqueue_style('stptp-bootstrap', 'https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css', array(), '5.3.3');
        wp_enqueue_script('stptp-bootstrap', 'https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js', array(), '5.3.3', true);
        // CSS leve do plugin
        wp_enqueue_style('stptp-admin', plugins_url('assets/admin.css', __FILE__), array(), STPTP_VERSION);
    }
}
add_action('admin_enqueue_scripts', 'stptp_admin_assets');

/** Helpers */
function stptp_money_to_float($v){
    $v = is_string($v) ? $v : (string)$v;
    $v = trim($v);
    // remove R$, pontos de milhar e troca vírgula por ponto
    $v = preg_replace('/[^0-9,.\-]/', '', $v);
    $v = str_replace('.', '', $v);
    $v = str_replace(',', '.', $v);
    return floatval($v);
}
function stptp_time_or_null($t){
    $t = trim((string)$t);
    return preg_match('/^\d{2}:\d{2}$/', $t) ? $t.':00' : null; // adiciona segundos
}
function stptp_html_money($n){
    return 'R$ ' . number_format((float)$n, 2, ',', '.');
}
function stptp_bool($x){ return $x ? 1 : 0; }

/**
 * Dashboard
 */
function stptp_render_dashboard(){
    ?>
    <div class="wrap">
      <h1 class="wp-heading-inline">Transporte</h1>
      <hr class="wp-header-end">
      <div class="container-fluid my-3">
        <div class="row g-3">
          <div class="col-12 col-md-6">
            <div class="card shadow-sm">
              <div class="card-body">
                <h5 class="card-title mb-2">Bem-vindo</h5>
                <p class="text-muted mb-0">Use o menu <strong>Rotas</strong> para cadastrar Origem, Destino, valores e horários.</p>
              </div>
            </div>
          </div>
          <div class="col-12 col-md-6">
            <div class="card h-100 shadow-sm">
              <div class="card-body">
                <h6 class="card-subtitle text-muted">Versão</h6>
                <div class="display-6">1.1.0</div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
    <?php
}

/**
 * ROTAS — CRUD com Bootstrap
 */
function stptp_render_rotas_page(){
    if ( ! current_user_can('edit_posts') ){
        wp_die(__('Sem permissão para acessar esta página.', 'stptp'));
    }

    global $wpdb; $table = STPTP_TABLE_ROTAS;

    // Ações: salvar/atualizar
    if ( isset($_POST['stptp_action']) && $_POST['stptp_action'] === 'save_rota' ){
        check_admin_referer('stptp_rota_form', 'stptp_nonce');

        $id      = isset($_POST['id']) ? intval($_POST['id']) : 0;
        $origem  = sanitize_text_field(wp_unslash($_POST['origem'] ?? ''));
        $destino = sanitize_text_field(wp_unslash($_POST['destino'] ?? ''));
        $valor_passagem = stptp_money_to_float($_POST['valor_passagem'] ?? '0');
        $tarifa  = stptp_money_to_float($_POST['tarifa'] ?? '0');
        $desconto = floatval($_POST['desconto_padrao'] ?? 0);
        if ($desconto < 0) $desconto = 0; if ($desconto > 100) $desconto = 100;
        $hora_partida = stptp_time_or_null($_POST['hora_partida'] ?? '');
        $hora_chegada = stptp_time_or_null($_POST['hora_chegada'] ?? '');
        $ativo = isset($_POST['ativo']) ? 1 : 0;

        $data = array(
            'origem' => $origem,
            'destino' => $destino,
            'valor_passagem' => $valor_passagem,
            'tarifa' => $tarifa,
            'desconto_padrao' => $desconto,
            'hora_partida' => $hora_partida,
            'hora_chegada' => $hora_chegada,
            'ativo' => $ativo,
        );
        $format = array('%s','%s','%f','%f','%f','%s','%s','%d');

        if ($id > 0){
            $wpdb->update($table, $data, array('id'=>$id), $format, array('%d'));
            $msg = 'Rota atualizada com sucesso.';
        } else {
            $wpdb->insert($table, $data, $format);
            $msg = 'Rota criada com sucesso.';
        }

        echo '<div class="notice notice-success is-dismissible"><p>'.esc_html($msg).'</p></div>';
    }

    // Ações: deletar
    if ( isset($_GET['action'], $_GET['_wpnonce']) && $_GET['action']==='delete' ){
        $id = intval($_GET['id'] ?? 0);
        if ( $id && wp_verify_nonce($_GET['_wpnonce'], 'stptp_del_'.$id) ){
            $wpdb->delete($table, array('id'=>$id), array('%d'));
            echo '<div class="notice notice-success is-dismissible"><p>Rota excluída.</p></div>';
        }
    }

    // Editar: busca a rota
    $edit = null;
    if ( isset($_GET['action']) && $_GET['action']==='edit' ){
        $id = intval($_GET['id'] ?? 0);
        if ($id){
            $edit = $wpdb->get_row( $wpdb->prepare("SELECT * FROM {$table} WHERE id=%d", $id) );
        }
    }

    // Lista de rotas
    $rotas = $wpdb->get_results("SELECT * FROM {$table} ORDER BY origem, destino");

    ?>
    <div class="wrap">
      <h1 class="wp-heading-inline">Rotas</h1>
      <hr class="wp-header-end">

      <div class="container-fluid py-3">
        <div class="row g-4">
          <!-- Formulário -->
          <div class="col-12 col-lg-5">
            <div class="card shadow-sm border-0">
              <div class="card-header bg-white">
                <strong><?php echo $edit ? 'Editar Rota' : 'Nova Rota'; ?></strong>
              </div>
              <div class="card-body">
                <form method="post" class="needs-validation" novalidate>
                  <?php wp_nonce_field('stptp_rota_form', 'stptp_nonce'); ?>
                  <input type="hidden" name="stptp_action" value="save_rota">
                  <input type="hidden" name="id" value="<?php echo esc_attr($edit->id ?? 0); ?>">

                  <div class="row g-3">
                    <div class="col-md-6">
                      <label class="form-label">Origem</label>
                      <input type="text" name="origem" class="form-control" required
                             value="<?php echo esc_attr($edit->origem ?? ''); ?>">
                      <div class="invalid-feedback">Informe a origem.</div>
                    </div>
                    <div class="col-md-6">
                      <label class="form-label">Destino</label>
                      <input type="text" name="destino" class="form-control" required
                             value="<?php echo esc_attr($edit->destino ?? ''); ?>">
                      <div class="invalid-feedback">Informe o destino.</div>
                    </div>

                    <div class="col-md-6">
                      <label class="form-label">Valor da Passagem</label>
                      <div class="input-group">
                        <span class="input-group-text">R$</span>
                        <input type="text" name="valor_passagem" class="form-control" placeholder="0,00" required
                               value="<?php echo esc_attr(isset($edit->valor_passagem) ? number_format((float)$edit->valor_passagem,2,',','.') : ''); ?>">
                      </div>
                      <small class="text-muted">Preço da passagem (sem tarifa).</small>
                    </div>
                    <div class="col-md-6">
                      <label class="form-label">Tarifa de Transporte</label>
                      <div class="input-group">
                        <span class="input-group-text">R$</span>
                        <input type="text" name="tarifa" class="form-control" placeholder="0,00" required
                               value="<?php echo esc_attr(isset($edit->tarifa) ? number_format((float)$edit->tarifa,2,',','.') : ''); ?>">
                      </div>
                      <small class="text-muted">Tarifa fixa aplicada à venda.</small>
                    </div>

                    <div class="col-md-6">
                      <label class="form-label">Desconto Padrão (%)</label>
                      <div class="input-group">
                        <input type="number" name="desconto_padrao" class="form-control" min="0" max="100" step="0.01"
                               value="<?php echo esc_attr(isset($edit->desconto_padrao) ? number_format((float)$edit->desconto_padrao,2,'.','') : '0'); ?>">
                        <span class="input-group-text">%</span>
                      </div>
                      <small class="text-muted">Aplicado por padrão nas vendas desta rota.</small>
                    </div>

                    <div class="col-md-6">
                      <label class="form-label">Ativo</label>
                      <div class="form-check form-switch">
                        <input class="form-check-input" type="checkbox" role="switch" name="ativo" id="ativoSwitch"
                               <?php echo (!isset($edit->ativo) || (int)$edit->ativo===1) ? 'checked' : ''; ?>>
                        <label class="form-check-label" for="ativoSwitch">Disponível para venda</label>
                      </div>
                    </div>

                    <div class="col-md-6">
                      <label class="form-label">Hora de Partida</label>
                      <input type="time" name="hora_partida" class="form-control"
                             value="<?php echo esc_attr(isset($edit->hora_partida) && $edit->hora_partida ? substr($edit->hora_partida,0,5) : ''); ?>">
                    </div>
                    <div class="col-md-6">
                      <label class="form-label">Hora de Chegada</label>
                      <input type="time" name="hora_chegada" class="form-control"
                             value="<?php echo esc_attr(isset($edit->hora_chegada) && $edit->hora_chegada ? substr($edit->hora_chegada,0,5) : ''); ?>">
                    </div>
                  </div>

                  <div class="mt-4 d-flex gap-2">
                    <button class="btn btn-primary" type="submit">
                      <?php echo $edit ? 'Salvar alterações' : 'Adicionar Rota'; ?>
                    </button>
                    <?php if ($edit): ?>
                      <a href="<?php echo esc_url( admin_url('admin.php?page=stptp-rotas') ); ?>" class="btn btn-outline-secondary">Cancelar</a>
                    <?php endif; ?>
                  </div>
                </form>
              </div>
            </div>
          </div>

          <!-- Lista -->
          <div class="col-12 col-lg-7">
            <div class="card shadow-sm border-0">
              <div class="card-header bg-white d-flex justify-content-between align-items-center">
                <strong>Rotas cadastradas</strong>
                <form method="get" class="m-0">
                  <?php foreach (array('page') as $p): ?>
                    <input type="hidden" name="<?php echo esc_attr($p); ?>" value="stptp-rotas">
                  <?php endforeach; ?>
                  <div class="input-group input-group-sm" style="width:280px;">
                    <input type="text" class="form-control" name="s" placeholder="Buscar por origem/destino" value="<?php echo esc_attr($_GET['s'] ?? ''); ?>">
                    <button class="btn btn-outline-secondary">Buscar</button>
                  </div>
                </form>
              </div>
              <div class="table-responsive">
                <table class="table align-middle mb-0">
                  <thead class="table-light">
                    <tr>
                      <th>#</th>
                      <th>Origem</th>
                      <th>Destino</th>
                      <th class="text-end">Passagem</th>
                      <th class="text-end">Tarifa</th>
                      <th class="text-end">Desc. (%)</th>
                      <th class="text-center">Partida</th>
                      <th class="text-center">Chegada</th>
                      <th class="text-center">Ativo</th>
                      <th></th>
                    </tr>
                  </thead>
                  <tbody>
                    <?php
                    // Filtro de busca simples
                    $needle = isset($_GET['s']) ? strtolower(sanitize_text_field($_GET['s'])) : '';
                    $count = 0;
                    foreach ($rotas as $r){
                        if ($needle){
                            $hay = strtolower($r->origem . ' ' . $r->destino);
                            if (strpos($hay, $needle) === false) continue;
                        }
                        $count++;
                        $edit_url = wp_nonce_url( admin_url('admin.php?page=stptp-rotas&action=edit&id='.$r->id), 'stptp_edit_'.$r->id );
                        $del_url  = wp_nonce_url( admin_url('admin.php?page=stptp-rotas&action=delete&id='.$r->id), 'stptp_del_'.$r->id );
                        echo '<tr>';
                        echo '<td>'.esc_html($r->id).'</td>';
                        echo '<td>'.esc_html($r->origem).'</td>';
                        echo '<td>'.esc_html($r->destino).'</td>';
                        echo '<td class="text-end">'.stptp_html_money($r->valor_passagem).'</td>';
                        echo '<td class="text-end">'.stptp_html_money($r->tarifa).'</td>';
                        echo '<td class="text-end">'.esc_html(number_format((float)$r->desconto_padrao,2,',','.')).'</td>';
                        echo '<td class="text-center">'.($r->hora_partida ? esc_html(substr($r->hora_partida,0,5)) : '-').'</td>';
                        echo '<td class="text-center">'.($r->hora_chegada ? esc_html(substr($r->hora_chegada,0,5)) : '-').'</td>';
                        echo '<td class="text-center">'.( $r->ativo ? "<span class='badge text-bg-success'>Sim</span>" : "<span class='badge text-bg-secondary'>Não</span>" ).'</td>';
                        echo '<td class="text-end">
                                <div class="btn-group btn-group-sm">
                                  <a class="btn btn-outline-primary" href="'.esc_url($edit_url).'">Editar</a>
                                  <a class="btn btn-outline-danger" href="'.esc_url($del_url).'" onclick="return confirm(\'Excluir esta rota?\');">Excluir</a>
                                </div>
                              </td>';
                        echo '</tr>';
                    }
                    if (!$count){
                        echo '<tr><td colspan="10" class="text-center text-muted py-4">Nenhuma rota encontrada.</td></tr>';
                    }
                    ?>
                  </tbody>
                </table>
              </div>
            </div>
          </div>

        </div>
      </div>
    </div>

    <script>
    (function(){
      // Bootstrap validation
      const forms = document.querySelectorAll('.needs-validation');
      Array.prototype.slice.call(forms).forEach(function(form){
        form.addEventListener('submit', function (event){
          if (!form.checkValidity()){
            event.preventDefault(); event.stopPropagation();
          }
          form.classList.add('was-validated');
        }, false);
      });
    })();
    </script>
    <?php
}

/* CSS inline para fallback caso o arquivo não carregue */
add_action('admin_head', function(){
  $screen = isset($_GET['page']) ? sanitize_text_field($_GET['page']) : '';
  if ($screen === 'stptp-rotas'){
    echo '<style>
      .stptp-badge{font-size:.75rem;padding:.35em .5rem;border-radius:.5rem}
    </style>';
  }
});
