<?php
/**
 * Template Name: Checkout Personalizado — Geratec (v33.3 — tarifa do Gratuito entra no Woo + desconto no TOTAL)
 *
 * Regras por bilhete (iguais às do painel):
 *  - Adulto:      Taxa = tarifa-base (já embutida no preço do produto); Passagem = (preço assento − taxa);   Valor = Passagem + Taxa (= preço do assento)
 *  - Criança 50%: Taxa = 0;                                             Passagem = (preço assento criança); Valor = Passagem
 *  - Gratuito:    Taxa = tarifa-base (via fee positiva no Woo);         Passagem = 0;                        Valor = Taxa
 *
 * Desconto (preview):
 *  - Sempre sobre o TOTAL (Passagens + Taxas), com clamp
 *  - Sincroniza com o Woo via fee negativa “Desconto (preview)” (sem duplicar cupons/fees negativas)
 */

defined('ABSPATH') || exit;

get_header();

/* =============================== HELPERS =============================== */
if (!function_exists('ger_norm')) {
  function ger_norm($s){
    $s = trim((string)$s);
    if ($s === '') return '';
    if (function_exists('transliterator_transliterate')) {
      $s = transliterator_transliterate('NFD; [:Nonspacing Mark:] Remove; NFC', $s);
    } elseif (function_exists('iconv')) {
      $s = @iconv('UTF-8','ASCII//TRANSLIT//IGNORE',$s);
    }
    return strtolower($s);
  }
}
if (!function_exists('ger_to_float')) {
  function ger_to_float($v){
    if (is_numeric($v)) return (float)$v;
    $v = (string)$v;
    $v = preg_replace('/[^\d,.\-]/','',$v);
    $v = preg_replace('/\.(?=\d{3}(?:\D|$))/','',$v); // milhar .
    $v = str_replace(',', '.', $v);                   // vírgula -> ponto
    return (float)$v;
  }
}

/* Trata esta página como checkout */
add_filter('woocommerce_is_checkout', function ($is_checkout) {
  if ($is_checkout) return true;
  if (is_page_template(basename(__FILE__))) return true;
  return false;
}, 1000);

/* precisa de pagamento se total > 0 */
add_filter('woocommerce_cart_needs_payment', function ($needs, $cart) {
  if ($cart && (float)$cart->get_total('edit') > 0) return true;
  return $needs;
}, 1000, 2);

/* Detectores de tipo */
if (!function_exists('ger_is_child_50')) {
  function ger_is_child_50($ci){
    if (isset($ci['ticket_type']) && (string)$ci['ticket_type'] === '1') return true;
    $cands = [];
    foreach (['tipo_assento','seat_label','category','categoria','label','tipo'] as $k){ if (!empty($ci[$k])) $cands[] = $ci[$k]; }
    if (!empty($ci['wbtm_seats']) && is_array($ci['wbtm_seats'])) {
      $r = reset($ci['wbtm_seats']);
      foreach (['ticket_name','label','fare_name','category','categoria'] as $rk){ if (!empty($r[$rk])) $cands[] = $r[$rk]; }
      if (isset($r['ticket_type']) && (string)$r['ticket_type'] === '1') return true;
    }
    foreach ($cands as $txt){
      $n = ger_norm($txt);
      $hasKid  = (bool) preg_match('/crian|crianca|criança|child|kids|infantil|menor/', $n);
      $hasHalf = (bool) preg_match('/50\s*%|\b50\b|meia|meia\-entrada|1\/2|metade|half/', $n);
      if ($hasKid && $hasHalf) return true;
    }
    return false;
  }
}
if (!function_exists('ger_is_free')) {
  function ger_is_free($ci){
    foreach (['tipo_assento','seat_label','category','categoria'] as $k){
      if (!empty($ci[$k]) && strpos(ger_norm($ci[$k]),'grat')!==false) return true;
    }
    if (!empty($ci['wbtm_seats']) && is_array($ci['wbtm_seats'])) {
      $r = reset($ci['wbtm_seats']);
      if (!empty($r['ticket_name']) && strpos(ger_norm($r['ticket_name']),'grat')!==false) return true;
    }
    return false;
  }
}

/* Preço do assento (produto/seat) e tarifa-base */
if (!function_exists('ger_ticket_assento')) {
  function ger_ticket_assento($ci){
    if (isset($ci['wbtm_ticket_price'])) return max(0.0, ger_to_float($ci['wbtm_ticket_price']));
    if (!empty($ci['wbtm_seats']) && is_array($ci['wbtm_seats'])) {
      $r = reset($ci['wbtm_seats']);
      if (!empty($r['ticket_price'])) return max(0.0, ger_to_float($r['ticket_price']));
    }
    $p = (!empty($ci['data']) && is_a($ci['data'],'WC_Product')) ? $ci['data'] : null;
    return $p ? (float) wc_get_price_to_display($p) : 0.0;
  }
}
if (!function_exists('ger_tarifa_base')) {
  function ger_tarifa_base($product_id, $ci = null){
    $keys = ['geratec_tarifa_unit','tarifa_transporte','_tarifa_transporte','_taxa_transporte','_ger_tarifa','_tarifa','tarifa'];
    foreach ($keys as $k){ if (is_array($ci) && isset($ci[$k]) && $ci[$k] !== '') return max(0.0, ger_to_float($ci[$k])); }
    foreach ($keys as $mk){ $v = get_post_meta((int)$product_id, $mk, true); if ($v !== '' && $v !== null) return max(0.0, ger_to_float($v)); }
    if (is_array($ci)){
      foreach (['wbtm_bus_id','bus_id','route_id'] as $idk){
        $rid = (int)($ci[$idk] ?? 0);
        if ($rid) {
          foreach ($keys as $mk){ $v = get_post_meta($rid, $mk, true); if ($v !== '' && $v !== null) return max(0.0, ger_to_float($v)); }
        }
      }
    }
    return 0.0;
  }
}

/* Cobrança (taxa) por tipo */
if (!function_exists('ger_tarifa_cobrada')) {
  function ger_tarifa_cobrada($ci){
    $base = ger_tarifa_base($ci['product_id'] ?? 0, $ci);
    if (ger_is_child_50($ci)) return 0.0; // Criança 50% não paga taxa
    return $base;                          // Adulto e Gratuito pagam tarifa-base
  }
}

/* Passagem (valor exibido) por tipo */
if (!function_exists('ger_passagem_valor')) {
  function ger_passagem_valor($ci){
    $ticket_assento = ger_ticket_assento($ci);
    $tarifa_cobr    = ger_tarifa_cobrada($ci);

    if (ger_is_child_50($ci)) {
      // Criança 50%: usa exatamente o preço do assento, sem taxa
      return max(0.0, round($ticket_assento, wc_get_price_decimals()));
    }
    if (ger_is_free($ci)) {
      // Gratuito: passagem 0 (a tarifa entra como fee positiva)
      return 0.0;
    }
    // Adulto: passagem = (preço assento − taxa)
    return max(0.0, round($ticket_assento - $tarifa_cobr, wc_get_price_decimals()));
  }
}

/* “Virtual rows” — 1 bilhete por linha */
if (!function_exists('ger_pick_seat_row')){
  function ger_pick_seat_row($ci, $idx){
    if (!empty($ci['wbtm_seats']) && is_array($ci['wbtm_seats'])) {
      $seats = array_values($ci['wbtm_seats']);
      if (isset($seats[$idx])) return $seats[$idx];
      if (isset($seats[0]))   return $seats[0];
    }
    return null;
  }
}
if (!function_exists('ger_pick_passenger_name')){
  function ger_pick_passenger_name($ci, $idx){
    if (!empty($ci['wbtm_passenger_info']) && is_array($ci['wbtm_passenger_info'])) {
      $pi = array_values($ci['wbtm_passenger_info']);
      if (!empty($pi[$idx]) && is_array($pi[$idx])) {
        $row = $pi[$idx];
        return $row['wbtm_full_name']['value'] ?? ($row['wbtm_full_name'] ?? ($row['name'] ?? ''));
      }
      if (!empty($pi[0]) && is_array($pi[0])) {
        $row = $pi[0];
        return $row['wbtm_full_name']['value'] ?? ($row['wbtm_full_name'] ?? ($row['name'] ?? ''));
      }
    }
    return '';
  }
}
if (!function_exists('ger_detect_units')){
  function ger_detect_units($ci){
    $qty    = max(1, (int)($ci['quantity'] ?? 1));
    $nSeats = (!empty($ci['wbtm_seats']) && is_array($ci['wbtm_seats'])) ? count($ci['wbtm_seats']) : 1;
    $nPass  = (!empty($ci['wbtm_passenger_info']) && is_array($ci['wbtm_passenger_info'])) ? count($ci['wbtm_passenger_info']) : 1;
    $nDecl  = !empty($ci['wbtm_seats_qty']) ? (int)$ci['wbtm_seats_qty'] : 1;
    return max($qty, $nSeats, $nPass, $nDecl);
  }
}
if (!function_exists('ger_virtual_rows')){
  function ger_virtual_rows($ci){
    $rows = [];
    $units = max(1, ger_detect_units($ci));
    for ($i=0; $i<$units; $i++){
      $row = $ci;
      $nome = ger_pick_passenger_name($ci, $i);
      if ($nome) $row['wbtm_passenger_name'] = $nome;
      $r = ger_pick_seat_row($ci, $i);
      if ($r) {
        if (!empty($r['ticket_name']))   $row['tipo_assento']      = $r['ticket_name'];
        if (!empty($r['seat_name']))     $row['seat_label']        = $r['seat_name'];
        if (isset($r['ticket_type']))    $row['ticket_type']       = $r['ticket_type'];
        if (!empty($r['date']))          $row['wbtm_date']         = $r['date'];
        if (!empty($r['ticket_price']))  $row['wbtm_ticket_price'] = ger_to_float($r['ticket_price']);
        $row['wbtm_seats'] = [$r];
      }
      $rows[] = $row;
    }
    return $rows;
  }
}
if (!function_exists('ger_count_virtual_rows')){
  function ger_count_virtual_rows($cart){
    if (!$cart) return 0;
    $c = 0;
    foreach ($cart->get_cart() as $ci) $c += count(ger_virtual_rows($ci));
    return $c;
  }
}

/* Somatórios (Passagens + Taxas) e Taxa só para Gratuito (fee positiva) */
if (!function_exists('ger_sum_passagens_taxas')){
  function ger_sum_passagens_taxas($cart){
    $sum_pass = 0.0; $sum_taxas = 0.0;
    foreach ($cart->get_cart() as $ci) {
      foreach (ger_virtual_rows($ci) as $row) {
        $sum_pass  += ger_passagem_valor($row);
        $sum_taxas += ger_tarifa_cobrada($row);
      }
    }
    return [$sum_pass, $sum_taxas];
  }
}
if (!function_exists('ger_sum_taxa_gratuito')){
  function ger_sum_taxa_gratuito($cart){
    $tot = 0.0;
    foreach ($cart->get_cart() as $ci) {
      foreach (ger_virtual_rows($ci) as $row) {
        if (ger_is_free($row)) {
          $tot += ger_tarifa_base($row['product_id'] ?? 0, $row);
        }
      }
    }
    return $tot;
  }
}

/* ================== FEES: Tarifa (Gratuito) + Desconto (preview) ================== */
add_action('woocommerce_cart_calculate_fees', function($cart){
  if (is_admin() && !defined('DOING_AJAX')) return;
  if (!$cart) return;

  /* 1) Fee POSITIVA: tarifa apenas para bilhetes Gratuito */
  $taxa_gratuito = ger_sum_taxa_gratuito($cart);
  if ($taxa_gratuito > 0) {
    // aparece como linha de tarifa; não é tributável
    $cart->add_fee(__('Tarifa (gratuito)', 'geratec'), $taxa_gratuito, false);
  }

  /* 2) Fee NEGATIVA: desconto preview sobre o TOTAL (Passagens + Taxas) */
  $pv = WC()->session ? WC()->session->get('ger_preview') : null;
  if ($pv && !empty($pv['set'])) {
    list($sum_pass, $sum_taxas) = ger_sum_passagens_taxas($cart);
    $baseTotal = max(0.0, $sum_pass + $sum_taxas); // este é o total “esperado” (exibe + Woo)

    $raw  = isset($pv['val']) ? (float)$pv['val'] : 0.0;
    $type = isset($pv['type']) ? (string)$pv['type'] : 'fixed';
    if ($raw > 0 && $baseTotal > 0) {
      $desired = preg_match('/%|percent/i', $type) ? ($baseTotal * ($raw/100.0)) : $raw;
      $desired = max(0.0, min($desired, $baseTotal));

      // já existentes (apenas negativos) para não duplicar
      $existing = 0.0;
      foreach ((array) $cart->fees_api()->get_fees() as $fee) {
        $amount = isset($fee->amount) ? (float)$fee->amount : 0.0;
        if ($amount < 0) $existing += abs($amount);
      }
      $existing += (float)$cart->get_discount_total();

      $effective = max(0.0, $desired - $existing);
      if ($effective > 0) {
        $cart->add_fee(__('Desconto (preview)', 'geratec'), -$effective, false);
      }
    }
  }
}, 999);

/* ================== ENDPOINT JSON (ger_json=1) ================== */
if (isset($_GET['ger_json']) && $_GET['ger_json'] === '1') {
  header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
  header('Pragma: no-cache');
  nocache_headers();

  if (function_exists('wc_load_cart')) wc_load_cart();

  $nonce_ok = !empty($_POST['nonce']) && wp_verify_nonce($_POST['nonce'], 'ger_summary');
  if (!$nonce_ok) { wp_send_json_error(['error'=>'bad_nonce'], 403); }

  // refletir método/endereço (para gateways/taxas)
  $pm      = isset($_POST['pm']) ? sanitize_text_field($_POST['pm']) : '';
  $country = isset($_POST['country']) ? sanitize_text_field($_POST['country']) : '';
  $state   = isset($_POST['state']) ? sanitize_text_field($_POST['state']) : '';
  $postcode= isset($_POST['postcode']) ? sanitize_text_field($_POST['postcode']) : '';
  $city    = isset($_POST['city']) ? sanitize_text_field($_POST['city']) : '';
  if ($pm) { $_POST['payment_method'] = $pm; WC()->session->set('chosen_payment_method', $pm); }
  if (WC()->customer) {
    if ($country) { WC()->customer->set_billing_country($country); WC()->customer->set_shipping_country($country); }
    if ($state)   { WC()->customer->set_billing_state($state);     WC()->customer->set_shipping_state($state); }
    if ($postcode){ WC()->customer->set_billing_postcode($postcode); WC()->customer->set_shipping_postcode($postcode); }
    if ($city)    { WC()->customer->set_billing_city($city);       WC()->customer->set_shipping_city($city); }
    WC()->customer->save();
  }

  $cart = WC()->cart;
  if ($cart) $cart->calculate_totals();

  list($sum_pass, $sum_taxas) = ger_sum_passagens_taxas($cart);
  $baseTotal = max(0.0, $sum_pass + $sum_taxas);

  // Preview do desconto (campo opcional)
  $preview_set = array_key_exists('wa_discount', $_POST); // mesmo "0"
  $op_val  = $preview_set ? ger_to_float($_POST['wa_discount']) : 0.0;
  $op_type = isset($_POST['wa_discount_type']) ? sanitize_text_field($_POST['wa_discount_type']) : 'fixed';

  // Persistir em sessão p/ hook da fee
  if (WC()->session) {
    if ($preview_set && $op_val > 0) {
      WC()->session->set('ger_preview', ['set'=>true, 'val'=>$op_val, 'type'=>$op_type]);
    } else {
      WC()->session->set('ger_preview', ['set'=>false, 'val'=>0, 'type'=>'fixed']);
    }
  }

  // cálculo do preview a exibir
  $preview_disc = 0.0;
  if ($preview_set && $op_val > 0 && $baseTotal > 0) {
    $preview_disc = preg_match('/%|percent/i', (string)$op_type) ? ($baseTotal * ($op_val/100.0)) : $op_val;
    $preview_disc = max(0.0, min($preview_disc, $baseTotal));
  }

  // descontos existentes (cupom + fees negativas)
  $ext_desc = 0.0;
  foreach ((array) $cart->fees_api()->get_fees() as $fee) {
    $amount = isset($fee->amount) ? (float)$fee->amount : 0.0;
    if ($amount < 0) $ext_desc += abs($amount);
  }
  $coupon_desc   = (float)$cart->get_discount_total();
  $all_desc_calc = min($ext_desc + $coupon_desc, $baseTotal);

  $all_desc_display = $preview_set ? $preview_disc : $all_desc_calc;

  $items_count  = $cart ? ger_count_virtual_rows($cart) : 0;
  $total_final  = max(0, $baseTotal - $all_desc_display);

  wp_send_json_success([
    'sum_valores_html' => wc_price($sum_pass),        // Passagens
    'sum_tarifas_html' => wc_price($sum_taxas),      // Taxas (adulto + gratuito)
    'desconto_html'    => wc_price($all_desc_display * -1),
    'total_html'       => wc_price($total_final),
    'items_count'      => $items_count,
    'items_label'      => sprintf('(%d %s)', $items_count, $items_count===1?'bilhete':'bilhetes'),
  ]);
  exit;
}

/* ================== RENDER ================== */
if (function_exists('wc_load_cart')) wc_load_cart();
if (!class_exists('WooCommerce')) { echo '<div class="woocommerce"><p>WooCommerce não está ativo.</p></div>'; get_footer(); exit; }

$cart = WC()->cart;
if (!$cart || $cart->is_empty()) { echo '<div class="woocommerce"><p>Seu carrinho está vazio.</p></div>'; get_footer(); exit; }

$cart->calculate_totals();
$checkout = WC()->checkout();
if (!$checkout->is_registration_enabled() && $checkout->is_registration_required() && !is_user_logged_in()) {
  echo '<div class="woocommerce"><p>'.esc_html__('Você deve entrar para finalizar a compra.', 'woocommerce').'</p></div>'; get_footer(); exit;
}
$nonce_summary = wp_create_nonce('ger_summary');

/* Totais iniciais (mesma lógica) */
$items_count_init = ger_count_virtual_rows($cart);
list($sum_pass_init, $sum_taxas_init) = ger_sum_passagens_taxas($cart);
$baseTotal_init   = max(0.0, $sum_pass_init + $sum_taxas_init);

// já considera cupons/fees negativas
$ext_desc_init = 0.0;
foreach ((array) $cart->fees_api()->get_fees() as $fee) {
  $amount = isset($fee->amount) ? (float)$fee->amount : 0.0;
  if ($amount < 0) $ext_desc_init += abs($amount);
}
$coupon_desc_init = (float)$cart->get_discount_total();
$all_desc_init    = min($ext_desc_init + $coupon_desc_init, $baseTotal_init);
$total_final_init = max(0, $baseTotal_init - $all_desc_init);
?>
<style>
.gerWrap *{box-sizing:border-box}
.gerWrap{--gap:18px;--radius:16px;--bd:#e7e7e7;--muted:#666;--soft:#f7f7f9;--shadow:0 4px 14px rgba(0,0,0,.06);
  max-width:1200px;margin:20px auto;padding:0 14px;color:#222;font-size:15px;line-height:1.45}
.gerGrid{display:grid;gap:var(--gap);grid-template-columns:1fr}
@media(min-width:992px){.gerGrid{grid-template-columns:1.15fr .85fr;align-items:start}}
.card{background:#fff;border:1px solid var(--bd);border-radius:16px;box-shadow:0 4px 14px rgba(0,0,0,.06)}
.pad{padding:16px 18px}
.hd{display:flex;justify-content:space-between;align-items:baseline;gap:12px;margin-bottom:8px}
.tt{font-size:18px;font-weight:800}
.sb{color:#777;font-size:13px}
.rItem{display:grid;grid-template-columns:1fr auto;gap:10px 16px;padding:14px 0;border-top:1px dashed #ececec}
.rItem:first-of-type{border-top:none}
.rTitle{font-size:15px;font-weight:700;color:#111;word-break:break-word}
.rMeta{margin-top:6px;color:#555;font-size:12.75px}
.rMeta .dot{margin:0 8px;color:#ccc}
.rPrice{text-align:right;font-size:13.5px;line-height:1.45}
.rPrice b{font-size:14px}
.rTotals{display:grid;gap:8px;border-top:1px solid #eee;padding-top:12px;margin-top:8px}
.rowTot{display:flex;justify-content:space-between;gap:10px}
.rowTot .lbl{color:#777;font-size:13px}
.rowTot .val{font-weight:800;font-size:16px}
.rowTot.desconto .val{color:#c0392b}
.rowTot.total .val{font-size:18px}
.pay{position:sticky;top:14px;z-index:2}
.payInner{padding:14px 16px}
.gerWrap .wc_payment_methods li.wc_payment_method{border:1px solid var(--bd);border-radius:12px;padding:10px 12px;background:#fff}
.gerWrap #place_order{margin-top:10px;border-radius:12px;padding:12px 18px;font-weight:800}
@media(max-width:991.98px){.pay{position:static}.rPrice{text-align:left}}
</style>

<div class="gerWrap" id="gerWrap" data-json-url="<?php echo esc_url( add_query_arg('ger_json','1', get_permalink()) ); ?>" data-nonce="<?php echo esc_attr($nonce_summary); ?>">
  <?php wc_print_notices(); ?>

  <form name="checkout" method="post" class="checkout woocommerce-checkout" action="<?php echo esc_url( wc_get_checkout_url() ); ?>" enctype="multipart/form-data">
    <?php do_action('woocommerce_before_checkout_form', $checkout); ?>

    <div class="gerGrid">
      <!-- ESQUERDA -->
      <div class="colL card pad">
        <div class="hd">
          <h3 class="tt">
            Resumo da reserva
            <span id="itemsCount" class="sb" style="margin-left:8px">
              (<?php echo esc_html($items_count_init); ?> <?php echo $items_count_init===1?'bilhete':'bilhetes'; ?>)
            </span>
          </h3>
          <p class="sb">Passagem (valor) + Taxa = Valor do bilhete</p>
        </div>

        <?php foreach ($cart->get_cart() as $ci):
          $product = !empty($ci['data']) && is_a($ci['data'],'WC_Product') ? $ci['data'] : null;
          if (!$product) continue;

          foreach (ger_virtual_rows($ci) as $row):
            $nome        = $product->get_name();
            $is_child    = ger_is_child_50($row);
            $is_free     = ger_is_free($row);
            $taxa_cobr   = ger_tarifa_cobrada($row);
            $pass_val    = ger_passagem_valor($row);
            $ticket_ass  = ger_ticket_assento($row);

            // Data/Hora sem duplicar
            $dp = [];
            $dd = trim((string)($row['wbtm_date'] ?? ''));
            $hh = trim((string)($row['wbtm_bp_time'] ?? ''));
            if ($dd !== '') $dp[] = $dd;
            if ($hh !== '' && $hh !== $dd) $dp[] = $hh;
            $dt = implode(' ', array_unique($dp));

            $parts = [];
            if (!empty($row['wbtm_passenger_name'])) $parts[] = '<strong>Passageiro:</strong> '.esc_html($row['wbtm_passenger_name']);
            if (!empty($row['tipo_assento']))        $parts[] = '<strong>Tipo:</strong> '.esc_html($row['tipo_assento']);
            if (!empty($dt))                         $parts[] = '<strong>Data/Horário:</strong> '.esc_html($dt);
            if (!empty($row['seat_label']))          $parts[] = '<strong>Assento:</strong> '.esc_html($row['seat_label']);
        ?>
          <div class="rItem">
            <div>
              <div class="rTitle">
                <?php echo esc_html($nome); ?>
                <?php if($is_child) echo ' — <em style="font-size:12px;color:#999">Criança 50%</em>'; ?>
                <?php if($is_free)  echo ' — <em style="font-size:12px;color:#999">Gratuito</em>'; ?>
              </div>
              <?php if ($parts): ?><div class="rMeta"><?php echo implode('<span class="dot">•</span>', $parts); ?></div><?php endif; ?>
            </div>
            <div class="rPrice">
              <?php
                echo 'Taxa = <b>'.wc_price($taxa_cobr).'</b><br>';
                echo 'Passagem = <b>'.wc_price($pass_val).'</b><br>';
                echo '<b>Valor '.wc_price($pass_val + $taxa_cobr).'</b>';
              ?>
            </div>
          </div>
        <?php endforeach; endforeach; ?>

        <!-- TOTAIS -->
        <div class="rTotals" id="rTotals">
          <div class="rowTot"><span class="lbl">Bilhetes</span><span class="val" id="sumItems"><?php echo esc_html($items_count_init); ?></span></div>
          <div class="rowTot"><span class="lbl">Subtotal (Passagens)</span><span class="val" id="sumValor"><?php echo wc_price($sum_pass_init); ?></span></div>
          <div class="rowTot"><span class="lbl">Taxas</span><span class="val" id="sumTarifas"><?php echo wc_price($sum_taxas_init); ?></span></div>
          <div class="rowTot desconto"><span class="lbl">Descontos</span><span class="val" id="sumDesc"><?php echo wc_price(($all_desc_init ?? 0) * -1); ?></span></div>
          <div class="rowTot total"><span class="lbl"><strong>Total</strong></span><span class="val" id="sumTotal"><?php echo wc_price($total_final_init); ?></span></div>
        </div>
      </div>

      <!-- DIREITA: pagamento -->
      <div class="colR card pay">
        <div class="payInner">
          <h3 class="tt">Formas de pagamento</h3>
          <?php woocommerce_checkout_payment(); ?>
          <?php
            $needs = WC()->cart && WC()->cart->needs_payment();
            $gws   = WC()->payment_gateways()->get_available_payment_gateways();
            if ($needs && empty($gws)) {
              echo '<div style="margin-top:10px;background:#fff7e6;border:1px dashed #f1c40f;border-radius:10px;padding:10px 12px;font-size:13px">';
              echo '<strong>Nenhuma forma de pagamento disponível.</strong><br>Preencha País/UF/CEP e verifique WooCommerce → Pagamentos. O total deve ser maior que zero.';
              echo '</div>';
            }
          ?>
          <!-- Campos de desconto (preview) - opcional -->
           
        </div>
      </div>
    </div>

    <?php do_action('woocommerce_after_checkout_form', $checkout); ?>
  </form>
</div>

<script>
(function waitForJQ(){
  if (!window.jQuery) { setTimeout(waitForJQ, 40); return; }
  jQuery(function($){
    const $wrap   = $('#gerWrap');
    const jsonUrl = ($wrap.data('json-url') || '').toString();
    const nonce   = ($wrap.data('nonce')    || '').toString();

    // anti duplo-submit
    (function lockSubmit(){
      const $form = $('form.checkout'); let locked = false;
      $form.on('submit', function(e){
        if (locked) { e.preventDefault(); e.stopImmediatePropagation(); return false; }
        locked = true;
        const $btn = $('#place_order');
        if ($btn.length){ $btn.data('orig-text', $btn.text()); $btn.prop('disabled', true).text('Processando...'); }
      });
      $(document).on('checkout_error', function(){
        locked = false; const $btn = $('#place_order');
        if ($btn.length){ $btn.prop('disabled', false).text($btn.data('orig-text') || 'Finalizar pedido'); }
      });
    })();

    // utils
    function debounce(fn,ms){ let t; return function(){ const c=this,a=arguments; clearTimeout(t); t=setTimeout(()=>fn.apply(c,a),ms); }; }
    function parseBR(v){ v=(v==null?'0':String(v)).replace(/[^\d,.\-]/g,''); v=v.replace(/\.(?=\d{3}(?:\D|$))/g,'').replace(',', '.'); const n=parseFloat(v); return isNaN(n)?0:n; }
    function currencySymbol(){ return ($('.woocommerce-Price-currencySymbol:first').text().trim() || 'R$'); }
    function priceHTML(value){
      const sym = currencySymbol();
      const sign = value < 0 ? '-' : '';
      const num  = Math.abs(value).toLocaleString('pt-BR',{minimumFractionDigits:2, maximumFractionDigits:2});
      return `<span class="woocommerce-Price-amount amount"><bdi>${sign}<span class="woocommerce-Price-currencySymbol">${sym}</span>&nbsp;${num}</bdi></span>`;
    }
    function readBRfromSpan(sel){ return parseBR($(sel).text()); }

    // ler campos do desconto
    function readDiscount(){
      const $v = $('#wa_discount, [name=wa_discount]').first();
      const raw = $v.length ? $v.val() : '0';
      let $t = $('#wa_discount_type, [name=wa_discount_type]').first();
      if (!$t.length) $t = $v.closest('div,fieldset,li').find('select').first();
      const ttxt = $t.length ? ($t.find('option:selected').text() || $t.val() || '') : '';
      const type = /%|percent/i.test(ttxt) ? '%' : 'fixed';
      return { raw: (raw===''?'0':String(raw)), type };
    }

    // preview instantâneo (desconto no TOTAL = Passagens + Taxas)
    function instantPreview(){
      const pass = readBRfromSpan('#sumValor');
      const taxas= readBRfromSpan('#sumTarifas');
      const d    = readDiscount();
      const entered  = parseBR(d.raw);
      const isPct    = d.type === '%';

      const baseTotal = Math.max(0, pass + taxas);
      let desconto = 0;
      if (entered > 0 && baseTotal > 0) desconto = isPct ? (baseTotal * (entered/100)) : entered;
      desconto = Math.max(0, Math.min(desconto, baseTotal));
      const total = Math.max(0, baseTotal - desconto);

      $('#sumDesc').html( priceHTML(-desconto) );
      $('#sumTotal').html( priceHTML(total) );
    }

    // aplicar resposta do endpoint
    function paint(data){
      if(!data) return;
      if (data.sum_valores_html) $('#sumValor').html(data.sum_valores_html);   // Passagens
      if (data.sum_tarifas_html) $('#sumTarifas').html(data.sum_tarifas_html); // Taxas
      if (data.desconto_html)    $('#sumDesc').html(data.desconto_html);
      if (data.total_html)       $('#sumTotal').html(data.total_html);
      if (typeof data.items_count !== 'undefined') $('#sumItems').text(data.items_count);
      if (typeof data.items_label !== 'undefined') $('#itemsCount').text(data.items_label);
    }

    // nosso AJAX: grava sessão p/ fee negativa e devolve resumo
    let busy = false;
    const fetchTotals = debounce(function(){
      if (!jsonUrl) return;
      if (busy) return; busy = true;
      const pm  = $('input[name=payment_method]:checked').val() || '';
      const d   = readDiscount();
      $.post(jsonUrl, {
        ger_json: 1, nonce,
        pm,
        country:  $('#billing_country').val() || '',
        state:    $('#billing_state').val() || '',
        postcode: $('#billing_postcode').val() || '',
        city:     $('#billing_city').val() || '',
        wa_discount: d.raw,
        wa_discount_type: d.type
      }).done(function(res){
        if (res && res.success) paint(res.data);
      }).always(function(){ busy = false; });
    }, 120);

    // força Woo recalcular (para as fees entrarem no total)
    const wooUpdateNow = debounce(function(){
      if (window.wc_checkout_form && typeof window.wc_checkout_form.update_checkout === 'function'){
        window.wc_checkout_form.update_checkout();
      } else {
        $(document.body).trigger('update_checkout');
      }
    }, 20);

    // eventos
    $(document).ready(function(){
      instantPreview();
      wooUpdateNow();
      fetchTotals();
    });

    $(document).on('change click', 'input[name=payment_method]', function(){ instantPreview(); wooUpdateNow(); fetchTotals(); });
    $(document).on('change keyup paste', '#billing_country,#billing_state,#billing_postcode,#billing_city', function(){ instantPreview(); wooUpdateNow(); fetchTotals(); });
    $(document).on('input keyup paste change blur', '#wa_discount,[name=wa_discount],#wa_discount_type,[name=wa_discount_type]', function(){
      instantPreview(); wooUpdateNow(); fetchTotals();
    });
  });
})();
</script>







<script>
jQuery(function($){
  const $form = $('form.checkout');
  function ensureMirror(){
    const $orig = $('#wa_discount'); if (!$orig.length || !$form.length) return;
    if (!$form.find('#wa_discount_mirror').length){
      $form.append('<input type="hidden" id="wa_discount_mirror" name="wa_discount" value="0">');
    }
    $('#wa_discount_mirror').val($orig.val());
  }
  $(document).on('input change', '#wa_discount, #wa_discount_type', function(){
    ensureMirror();
    if (window.wc_checkout_form && typeof window.wc_checkout_form.update_checkout === 'function'){
      window.wc_checkout_form.update_checkout();
    } else {
      $(document.body).trigger('update_checkout');
    }
  });
  $(document).ready(ensureMirror);
});
</script>



<?php get_footer(); ?>
