<?php
/**
 * Template Name: Comprovante — Geratec (Top: por passageiro + criança/gratuito + RawBT/QZ)
 * Description: Comprovante com impressão RawBT (Android) e QZ Tray (desktop). Resumo por passageiro.
 */
defined('ABSPATH') || exit;

get_header();

/* =========================
 * CONFIG
 * ========================= */
/** true  = ticket_price do plugin já é TOTAL (valor+tarifa) → subtrai tarifa-base para achar o valor
 *  false = ticket_price é somente VALOR (sem tarifa)
 */
if (!defined('GER_TICKET_PRICE_INCLUDES_TARIFA')) {
  define('GER_TICKET_PRICE_INCLUDES_TARIFA', true);
}

/** Formato da numeração do bilhete (exibe quando há 2+) — placeholders: %CURR%, %TOTAL% */
if (!defined('GER_TICKET_NUM_FORMAT')) {
  define('GER_TICKET_NUM_FORMAT', '%CURR%/%TOTAL%'); // ex.: "1/2"
}

/** Mostrar linha “Tarifa (unit.)” para criança? false = oculta (mais limpo) */
if (!defined('GER_SHOW_TARIFA_ROW_FOR_CHILD')) {
  define('GER_SHOW_TARIFA_ROW_FOR_CHILD', false);
}

/* =========================
 * Segurança básica
 * ========================= */
$order_id = absint($_GET['order_id'] ?? (get_query_var('order-received') ?: 0));
$key_in   = isset($_GET['key']) ? sanitize_text_field($_GET['key']) : '';
$order    = $order_id ? wc_get_order($order_id) : null;
if (!$order) wp_die('Pedido inválido.');

$owner_ok = is_user_logged_in() && (get_current_user_id() === (int) $order->get_user_id());
$key_ok   = $key_in && hash_equals($order->get_order_key(), $key_in);
if (!$owner_ok && !$key_ok) wp_die('Você não tem permissão para ver este comprovante.');

$brand    = get_bloginfo('name');
$currency = $order->get_currency();

/* =========================
 * Helpers
 * ========================= */
$meta_first = function($item, array $keys, $fallback = '') {
  foreach ($keys as $k) { $v = $item->get_meta($k); if ($v !== '' && $v !== null) return $v; }
  return $fallback;
};
$fmt_dt = function($str){
  if (!$str) return '';
  $ts = strtotime($str);
  return $ts ? date_i18n('d/m/Y H:i', $ts) : $str;
};
$to_float = function($v){
  if (is_numeric($v)) return (float)$v;
  $v = (string)$v;
  $v = preg_replace('/[^\d,.\-]/','',$v);
  $v = preg_replace('/\.(?=\d{3}(?:\D|$))/','',$v);
  $v = str_replace(',', '.', $v);
  return (float)$v;
};

/** Passageiros (nome/telefone) por item */
$get_passengers = function(WC_Order_Item $item, WC_Order $order){
  $out = [];
  $pi = $item->get_meta('_wbtm_passenger_info');
  if (is_array($pi) && $pi) {
    foreach ($pi as $row) {
      $nome = $row['wbtm_full_name']['value'] ?? ($row['wbtm_full_name'] ?? '');
      $fone = $row['wbtm_reg_phone']['value'] ?? ($row['wbtm_reg_phone'] ?? '');
      $nome = trim((string)$nome); $fone = trim((string)$fone);
      $out[] = ['nome' => $nome ?: (trim($order->get_billing_first_name().' '.$order->get_billing_last_name()) ?: 'Passageiro'), 'fone' => $fone];
    }
  }
  if (!$out) $out[] = ['nome'=>trim($order->get_billing_first_name().' '.$order->get_billing_last_name()) ?: 'Passageiro', 'fone'=>''];
  return $out;
};

/** Tickets/assentos por item */
$get_ticket_rows = function(WC_Order_Item $item){
  $rows = [];
  $ti = $item->get_meta('_wbtm_ticket_info');
  if (is_array($ti) && $ti) {
    foreach ($ti as $r) {
      $qty  = (int)($r['ticket_qty'] ?? 1);
      $seat = $r['seat_name'] ?? '';
      for ($i=0; $i<max(1,$qty); $i++) {
        $rows[] = [
          'seat'   => trim((string)$seat),
          'tarifa' => trim((string)($r['ticket_name'] ?? '—')), // “Adulto”, “Criança 50%”, “Gratuito”, etc.
          'price'  => (string)($r['ticket_price'] ?? ''),       // TOTAL (valor+tarifa) ou VALOR (config)
          'date'   => (string)($r['date'] ?? '')
        ];
      }
    }
  }
  if (!$rows) {
    foreach ($item->get_meta_data() as $m) {
      $d = $m->get_data();
      if (isset($d['key']) && mb_strtolower($d['key']) === 'assento') {
        $rows[] = ['seat'=>trim((string)$d['value']), 'tarifa'=>'—', 'price'=>'', 'date'=>''];
      }
    }
  }
  return $rows;
};

/** Serviços extras a partir de _extra_services */
$get_extra_services = function(WC_Order_Item $item) use ($to_float){
  $out = []; $ex = $item->get_meta('_extra_services'); if (empty($ex)) return $out;
  $push = function($title,$qty,$price) use (&$out,$to_float){
    $title = trim((string)$title); if ($title==='') return;
    $qty   = (int)($qty ?: 1); $price = $to_float($price);
    $out[] = ['title'=>$title,'qty'=>$qty,'price'=>$price,'total'=>$price*$qty];
  };
  if (is_array($ex)) {
    foreach ($ex as $k=>$v) {
      if (is_array($v)) $push($v['name'] ?? ($v['title'] ?? ($v['service'] ?? $k)), $v['qty'] ?? ($v['quantity'] ?? 1), $v['price'] ?? ($v['valor'] ?? ($v['amount'] ?? 0)));
      else $push($v,1,0);
    }
  } else foreach (preg_split('/\s*,\s*/', (string)$ex) as $s) $push($s,1,0);
  return $out;
};

/** Tarifa unitária “base” (sem fator) — tenta item, produto, rota/ônibus */
$get_tarifa_unit_for_item = function(WC_Order_Item $item) use ($to_float){
  foreach (['geratec_tarifa_unit','_geratec_tarifa_unit','tarifa_transporte','_tarifa_transporte'] as $k) {
    $v = $item->get_meta($k);
    if ($v !== '' && $v !== null) return max(0.0, $to_float($v));
  }
  $pid = $item->get_product_id();
  if ($pid) {
    $v = get_post_meta($pid, '_tarifa_transporte', true);
    if ($v !== '' && $v !== null) return max(0.0, $to_float($v));
  }
  foreach (['_wbtm_bus_id','_bus_id','route_id'] as $mk) {
    $id = (int) $item->get_meta($mk);
    if ($id) {
      $vv = get_post_meta($id, '_tarifa_transporte', true);
      if ($vv !== '' && $vv !== null) return max(0.0, $to_float($vv));
    }
  }
  return null; // se não achar, fica null (fallback 0 mais abaixo)
};

/* ===== Categoria: detectar e aplicar fatores ===== */
if (!function_exists('ger_norm')) {
  function ger_norm($s){
    $s = mb_strtolower(trim((string)$s));
    $s = function_exists('iconv') ? @iconv('UTF-8','ASCII//TRANSLIT//IGNORE',$s) : $s;
    return preg_replace('/\s+/', ' ', $s);
  }
}
if (!function_exists('ger_detect_category')) {
  function ger_detect_category(array $sources){
    $hay = ger_norm(implode(' ', array_filter(array_map('strval', $sources))));
    if ($hay === '') return 'adult';
    if (preg_match('/grat|cortesia|isento|free/', $hay))  return 'free';   // Gratuito
    if (preg_match('/crian|crianca|criança|child|infantil/', $hay) && preg_match('/50\s*%|\b50\b/', $hay)) return 'child'; // Criança 50%
    if (preg_match('/crian|crianca|criança|child|infantil|meia/', $hay)) return 'child';
    return 'adult';
  }
}
if (!function_exists('ger_factors_by_category')) {
  function ger_factors_by_category($cat){
    switch ($cat) {
      case 'child': return [1.0, 0.0]; // *** Criança: 100% do VALOR, 0% da TARIFA ***
      case 'free':  return [0.0, 1.0]; // Gratuito: 0% VALOR, 100% TARIFA
      default:      return [1.0, 1.0]; // Adulto: 100% dos dois
    }
  }
}
if (!function_exists('ger_category_label')) {
  function ger_category_label($cat){
    switch($cat){
      case 'child': return 'Criança 50%';
      case 'free':  return 'Gratuito';
      default:      return 'Adulto';
    }
  }
}

/* =========================
 * Fees do pedido (p/ desconto/extra)
 * ========================= */
$desconto_operador = 0.0;
$outras_fees = 0.0;
foreach ($order->get_items('fee') as $fee_item) {
  $name   = (string)$fee_item->get_name();
  $amount = (float)$fee_item->get_total();
  if ($amount < 0 && (preg_match('/desconto/i', $name) || preg_match('/operador/i', $name) || stripos($name,'Desconto do Operador (GER)')!==false)) {
    $desconto_operador += abs($amount);
  } else {
    $outras_fees += $amount;
  }
}

/** Quantidade total de bilhetes (para numeração e fallback de tarifa) */
$total_tickets = 0;
foreach ($order->get_items() as $it) {
  $pass = $get_passengers($it, $order); $trs  = $get_ticket_rows($it);
  $qty_meta = (int)$it->get_meta('_wbtm_qty');
  $total_tickets += max(count($pass), count($trs), (int)$it->get_quantity(), $qty_meta, 1);
}
$printed = 0;

/** Helper numeração formatada */
$format_ticket_num = function($curr, $tot){
  $s = GER_TICKET_NUM_FORMAT;
  $s = str_replace('%CURR%', (string)$curr, $s);
  $s = str_replace('%TOTAL%', (string)$tot, $s);
  return trim($s);
};

/* =========================
 * Resumo por passageiro (N linhas) + totais calculados
 * ========================= */
$bilhetes = [];
$subtotal_valor_calc = 0.0; // soma dos VALORES por bilhete
$tarifas_total_calc  = 0.0; // soma das TARIFAS por bilhete

foreach ($order->get_items() as $item_id => $item) {
  $passengers = $get_passengers($item, $order);
  $tickets    = $get_ticket_rows($item);

  $qty_meta   = (int)$item->get_meta('_wbtm_qty');
  $here       = max(count($passengers), count($tickets), (int)$item->get_quantity(), $qty_meta, 1);

  // Base de rateio (valor sem tarifa) e tarifa base por item
  $line_subtotal   = (float)$item->get_subtotal(); // valor (sem tarifa) — fallback se precisar
  $rateio_valor    = $here > 0 ? ($line_subtotal / $here) : $line_subtotal;

  $tarifa_item_meta = $get_tarifa_unit_for_item($item);
  $tarifa_fallback  = 0.0;

  for ($i=0; $i<$here; $i++) {
    $p  = $passengers[$i] ?? end($passengers);
    $tr = $tickets[$i]    ?? ['seat'=>'','tarifa'=>'—','price'=>'','date'=>''];

    $seat            = trim((string)($tr['seat'] ?? ''));
    $tarifa_cat_meta = $meta_first($item, ['Tipo de assento', '_wbtm_ticket_type', '_ticket_type'], '');
    $cat             = ger_detect_category([$tr['tarifa'] ?? '', $tarifa_cat_meta, $seat]);
    [$f_valor, $f_tarifa] = ger_factors_by_category($cat);

    // Valor base (sem tarifa) — **EXCEÇÃO CRIANÇA**:
    if (($tr['price'] ?? '') !== '') {
      $ticket_price = $to_float($tr['price']);
      if (GER_TICKET_PRICE_INCLUDES_TARIFA) {
        $tarifa_base_ref = ($tarifa_item_meta !== null) ? (float)$tarifa_item_meta : (float)$tarifa_fallback;

        // Criança 50% NÃO subtrai tarifa do preço do ticket
        if ($cat === 'child') {
          $valor_base = $ticket_price;         // já é o valor final para criança
        } else {
          $valor_base = max(0, $ticket_price - $tarifa_base_ref);
        }
      } else {
        $valor_base = $ticket_price;           // já é valor puro
      }
    } else {
      $valor_base = $rateio_valor;             // fallback
    }

    // Tarifa base e aplicação dos fatores
    $tarifa_base         = ($tarifa_item_meta !== null) ? (float)$tarifa_item_meta : (float)$tarifa_fallback;
    $valor_unit          = round($valor_base  * $f_valor,  wc_get_price_decimals());
    $tarifa_unit_ticket  = round($tarifa_base * $f_tarifa, wc_get_price_decimals());
    if ($cat === 'child') $tarifa_unit_ticket = 0.0; // garantia extra

    $total_bilhete = max(0, round($valor_unit + $tarifa_unit_ticket, wc_get_price_decimals()));

    // acumula totais calculados
    $subtotal_valor_calc += $valor_unit;
    $tarifas_total_calc  += $tarifa_unit_ticket;

    $bilhetes[] = [
      'nome'     => (string)($p['nome'] ?? 'Passageiro'),
      'categoria'=> ger_category_label($cat),
      'assento'  => $seat,
      'valor'    => $valor_unit,
      'tarifa'   => $tarifa_unit_ticket,
      'total'    => $total_bilhete,
    ];
  }
}

/* ---------- Totais finais para o “Resumo financeiro” (alinhados ao checkout) ---------- */
$subtotal_valor = $subtotal_valor_calc; // soma dos valores por bilhete
$tarifas_total  = $tarifas_total_calc;  // soma das tarifas por bilhete
// Total do resumo (informativo): valor + tarifas − desconto + outras fees
$total_resumo   = max(0, ($subtotal_valor + $tarifas_total) - $desconto_operador + $outras_fees);
?>
<style>
  body{margin:0;background:#f6f7fb;font-family:system-ui,Segoe UI,Roboto,Arial;color:#0f172a}
  .container{max-width:980px;margin:18px auto;padding:0 12px}
  .card{background:#fff;border:1px solid #e5e7eb;border-radius:14px;box-shadow:0 6px 18px rgba(0,0,0,.06);overflow:hidden}
  .card .hd{display:flex;justify-content:space-between;align-items:center;padding:12px 14px;border-bottom:1px solid #eef2f7}
  .card .bd{padding:12px 14px;display:grid;gap:10px}

  .toolbar{display:flex;gap:10px;flex-wrap:wrap}
  .btn{padding:10px 12px;border:0;border-radius:10px;font-weight:700;cursor:pointer}
  .btn[disabled]{opacity:.55;cursor:not-allowed}
  .btn-green{background:#22c55e;color:#053}
  .btn-blue{background:#0ea5e9;color:#fff}
  .btn-ghost{background:#fff;border:1px solid #e5e7eb}
  .status{font-size:13px;color:#475569;margin:6px 0}
  .dot{width:10px;height:10px;border-radius:999px;background:#cbd5e1;display:inline-block;margin-right:6px;vertical-align:middle}
  .ok{background:#22c55e}.warn{background:#f59e0b}.err{background:#ef4444}
  .qz-select{padding:10px;border:1px solid #e5e7eb;border-radius:10px;min-width:240px}

  /* Totais */
  .totals{margin:14px auto 6px; max-width:980px}
  .totals .row{display:flex;justify-content:space-between;align-items:center;padding:6px 0;border-bottom:1px dashed #e9eef5}
  .totals .row:last-child{border-bottom:0}
  .totals .lbl{font-size:14px;color:#475569}
  .totals .val{font-weight:800}
  .totals .desc{color:#c0392b}
  .totals .final{font-size:16px}

  /* Bilhetes preview 80mm */
  :root { --w: 80mm; }
  .ticket-wrap{width:var(--w); margin:16px auto; font-family:Arial,Helvetica,sans-serif; font-size:12px; line-height:1.35; color:#0f172a}
  .ticket{page-break-inside: avoid; background:#fff; border:1px solid #e5e7eb; border-radius:10px; padding:10px; margin:16px 0; box-shadow:0 6px 12px rgba(0,0,0,.06)}
  .t-head{ text-align:center; margin:4px 0 2px }
  .t-head h3{ margin:0; font-size:15px; letter-spacing:.2px }
  .badge-small{display:inline-block; padding:2px 8px; border-radius:999px; font-size:11px; color:#0369a1; background:#e0f2fe; border:1px solid #bae6fd; margin-top:4px}
  .hr{ border:none; border-top:1px dashed #94a3b8; margin:8px 0 }
  .t-meta{ display:grid; gap:3px; margin-top:2px }
  .t-meta div{ display:flex; justify-content:space-between; gap:6px }
  .t-meta strong{ color:#111827 }
  .t-sec{ margin-top:6px }
  .route{ display:grid; grid-template-columns:1fr 24px 1fr; align-items:center; gap:6px; margin:6px 0 6px }
  .route .dot{ width:7px; height:7px; border-radius:999px; background:#10b981; margin:0 auto }
  .route .line{ width:2px; height:20px; background:#e2e8f0; margin:0 auto }
  .route .col{ font-size:12px; color:#0f172a }
  .route .col b{ display:block; color:#111827 }
  .tbl{ width:100%; border-collapse:collapse }
  .tbl td{ padding:3px 0; border-bottom:1px dashed #e5e7eb; vertical-align:top }
  .tbl tr:last-child td{ border-bottom:0 }
  .tbl td:first-child{ width:46%; color:#334155 }
  .tbl td:last-child{ width:54%; color:#111827; text-align:right }

  .cut{ border:none; border-top:1px dashed #cbd5e1; margin:16px 0; position:relative }
  .cut:after{ content:' '; position:absolute; right:0; top:-10px; font-size:10px; color:#94a3b8 }

  /* Modal */
  .modal{position:fixed;inset:0;background:rgba(15,23,42,.55);display:none;align-items:center;justify-content:center;z-index:9999}
  .modal.open{display:flex}
  .modal .box{width:min(900px,96vw);background:#fff;border-radius:14px;box-shadow:0 20px 60px rgba(0,0,0,.25);overflow:hidden}
  .modal .hd{display:flex;justify-content:space-between;align-items:center;padding:12px 14px;border-bottom:1px solid #eef2f7}
  .modal .bd{padding:12px 14px}
  .tabs{display:flex;gap:8px;border-bottom:1px solid #eef2f7;margin-bottom:10px}
  .tab{padding:8px 12px;border-radius:10px 10px 0 0;background:#f8fafc;border:1px solid #eef2f7;border-bottom:0;cursor:pointer}
  .tab.active{background:#fff;border-color:#cfd8e3}
  .panel{display:none}
  .panel.active{display:block}
  .grid2{display:grid;gap:14px}
  @media(min-width:860px){.grid2{grid-template-columns:1fr 1fr}}
  .field{display:flex;flex-direction:column;gap:6px}
  .field label{font-size:13px;color:#475569}
  .field input[type="number"], .field select{padding:10px;border:1px solid #e5e7eb;border-radius:10px}

  /* Esconde elementos do tema (se necessário) */
  .bahon_page_content h2.bahon_single_post_title,
  .woocommerce-account .woocommerce-MyAccount-navigation ul { display:none }
  .p-70-60 { padding-top:0; padding-bottom:0 }
  body.page .bahon_page_content { background-color:#fff; padding:0; box-shadow:none }
</style>

<div class="container">
  <!-- Toolbar -->
  <div class="card">
    <div class="hd">
      <strong>Impressão</strong><span><?php echo esc_html($brand); ?></span>
    </div>
    <div class="bd">
      <div class="toolbar">
        <button id="btn-print-rawbt" class="btn btn-green">Imprimir (RawBT)</button>
        <button id="btn-print-qz"    class="btn btn-blue" disabled>Imprimir (QZ Tray)</button>
        <button id="btn-open-config" class="btn btn-ghost">Configurar</button>
      </div>
      <div class="status">
        <span class="dot" id="dot-rawbt"></span><span id="txt-rawbt">RawBT: aguardando Android</span> ·
        <span class="dot" id="dot-qz"></span><span id="txt-qz">QZ Tray: conectando…</span>
      </div>
    </div>
  </div>

  <!-- Resumo financeiro -->
  <div class="card totals">
    <div class="hd"><strong>Resumo financeiro do pedido</strong><span>#<?php echo esc_html($order->get_order_number()); ?></span></div>
    <div class="bd">
      <div class="row"><span class="lbl">Subtotal (valor)</span><span class="val"><?php echo wc_price($subtotal_valor, ['currency'=>$currency]); ?></span></div>
      <div class="row"><span class="lbl">Tarifas</span><span class="val"><?php echo wc_price($tarifas_total, ['currency'=>$currency]); ?></span></div>
      <div class="row"><span class="lbl">Desconto do operador</span><span class="val desc"><?php echo wc_price($desconto_operador * -1, ['currency'=>$currency]); ?></span></div>
      <?php if (abs($outras_fees) > 0.0001): ?>
        <div class="row"><span class="lbl">Outras taxas/descontos</span><span class="val"><?php echo wc_price($outras_fees, ['currency'=>$currency]); ?></span></div>
      <?php endif; ?>
      <div class="row"><span class="lbl"><strong>Total (valor+tarifas−desconto)</strong></span><span class="val final"><strong><?php echo wc_price($total_resumo, ['currency'=>$currency]); ?></strong></span></div>
      <div class="row"><span class="lbl">Total do pedido (Woo)</span><span class="val"><?php echo wp_kses_post($order->get_formatted_order_total()); ?></span></div>
    </div>
  </div>

  <!-- Resumo por passageiro -->
  <?php if (!empty($bilhetes)): ?>
    <div class="card totals">
      <div class="hd"><strong>Resumo por passageiro</strong><span><?php echo count($bilhetes); ?> bilhetes</span></div>
      <div class="bd">
        <?php foreach ($bilhetes as $b): ?>
          <div class="row">
            <span class="lbl">
              <?php echo esc_html($b['nome']); ?> — <?php echo esc_html($b['categoria']); ?>
              <?php if(!empty($b['assento'])) echo ' • Assento: '.esc_html($b['assento']); ?>
            </span>
            <span class="val"><?php echo wc_price($b['total'], ['currency'=>$currency]); ?></span>
          </div>
          <div class="row">
            <span class="lbl" style="padding-left:12px;">Valor / Tarifa</span>
            <span class="val">
              <?php echo wc_price($b['valor'],  ['currency'=>$currency]); ?>
              /
              <?php echo wc_price($b['tarifa'], ['currency'=>$currency]); ?>
            </span>
          </div>
        <?php endforeach; ?>
      </div>
    </div>
  <?php endif; ?>

  <!-- Bilhetes -->
  <div class="ticket-wrap" id="ticketWrap">
  <?php
  foreach ($order->get_items() as $item_id => $item):
    $name  = $item->get_name();

    $bp      = $meta_first($item, ['_wbtm_bp','Embarque']);
    $bp_time = $meta_first($item, ['_wbtm_bp_time']);
    $dp      = $meta_first($item, ['_wbtm_dp','Deixando cair','Destino']);
    $dp_time = $meta_first($item, ['_wbtm_dp_time']);

    $start_point = $meta_first($item, ['_wbtm_start_point']);
    $pickup      = $meta_first($item, ['_wbtm_pickup_point']);
    $dropoff     = $meta_first($item, ['_wbtm_drop_off_point']);

    $passengers = $get_passengers($item, $order);
    $tickets    = $get_ticket_rows($item);
    $extras     = $get_extra_services($item);

    $qty_meta   = (int)$item->get_meta('_wbtm_qty');
    $here       = max(count($passengers), count($tickets), (int)$item->get_quantity(), $qty_meta, 1);

    // Rateio de valor (subtotal) por bilhete, fallback se não houver ticket_price
    $line_subtotal = (float)$item->get_subtotal(); // valor (sem tarifa)
    $rateio_valor  = $here > 0 ? ($line_subtotal / $here) : $line_subtotal;

    // Tarifa “base” unitária preferencial via meta; fallback: 0
    $tarifa_item_meta = $get_tarifa_unit_for_item($item);
    $tarifa_fallback  = 0.0;

    for ($i=0; $i<$here; $i++):
      $printed++;
      $p  = $passengers[$i] ?? end($passengers);
      $tr = $tickets[$i]    ?? ['seat'=>'','tarifa'=>'—','price'=>'','date'=>''];

      $seat             = trim((string)($tr['seat'] ?? ''));
      $tarifa_cat_meta  = $meta_first($item, ['Tipo de assento', '_wbtm_ticket_type', '_ticket_type'], '');
      $tarifa_cat_text  = ($tr['tarifa'] ?? '') ?: $tarifa_cat_meta;

      // Detecta categoria com base em múltiplas fontes (tarifa, meta, assento)
      $cat = ger_detect_category([$tr['tarifa'] ?? '', $tarifa_cat_meta, $seat]);

      // fatores por categoria
      [$f_valor, $f_tarifa] = ger_factors_by_category($cat);

      // 1) base de valor (sem tarifa) — **EXCEÇÃO CRIANÇA**:
      if (($tr['price'] ?? '') !== '') {
        $ticket_price = $to_float($tr['price']);
        if (GER_TICKET_PRICE_INCLUDES_TARIFA) {
          $tarifa_base_ref = ($tarifa_item_meta !== null) ? (float)$tarifa_item_meta : (float)$tarifa_fallback;

          // Criança 50% NÃO subtrai tarifa do preço do ticket
          if ($cat === 'child') {
            $valor_base = $ticket_price;       // 75 permanece 75
          } else {
            $valor_base = max(0, $ticket_price - $tarifa_base_ref);
          }
        } else {
          $valor_base = $ticket_price;         // já é valor
        }
      } else {
        $valor_base = $rateio_valor;           // fallback
      }

      // 2) tarifa base unitária (sem fator)
      $tarifa_base = ($tarifa_item_meta !== null) ? (float)$tarifa_item_meta : (float)$tarifa_fallback;

      // 3) aplica fatores (Criança=0% tarifa; Gratuito=100% tarifa; Adulto=100%)
      $valor_unit         = round($valor_base  * $f_valor,  wc_get_price_decimals());
      $tarifa_unit_ticket = round($tarifa_base * $f_tarifa, wc_get_price_decimals());

      // Garantia extra: criança nunca paga tarifa
      if ($cat === 'child') {
        $tarifa_unit_ticket = 0.0;
      }

      // 4) total do bilhete
      $total_bilhete = max(0, round($valor_unit + $tarifa_unit_ticket, wc_get_price_decimals()));
  ?>
    <div class="ticket">
      <div class="t-head">
        <h3><?php echo esc_html("Passagens"); ?></h3>
        <?php if ($total_tickets > 1): ?>
          <div class="badge-small">
            <?php echo esc_html( $format_ticket_num($printed, $total_tickets) ); ?>
          </div>
        <?php endif; ?>
      </div>

      <hr class="hr">
      <div class="t-meta">
        <div><strong>Nº do pedido</strong><span><?php echo esc_html($order->get_order_number()); ?></span></div>
        <div><strong>Data</strong><span><?php echo esc_html(wc_format_datetime($order->get_date_created())); ?></span></div>
        <div><strong>Método</strong><span><?php echo esc_html($order->get_payment_method_title()); ?></span></div>
      </div>

      <div class="route">
        <div class="col"><b><?php echo esc_html($bp ?: ($start_point ?: 'Origem')); ?></b><?php echo $bp_time ? esc_html(' '.$fmt_dt($bp_time)) : ''; ?></div>
        <div><div class="dot"></div><div class="line"></div><div class="dot"></div></div>
        <div class="col" style="text-align:right"><b><?php echo esc_html($dp ?: 'Destino'); ?></b><?php echo $dp_time ? esc_html(' '.$fmt_dt($dp_time)) : ''; ?></div>
      </div>

      <div class="t-sec">
        <table class="tbl">
          <tr><td colspan="2"><strong><?php echo esc_html($name); ?></strong></td></tr>
          <tr><td>Passageiro</td><td><?php echo esc_html($p['nome']); ?></td></tr>
          <?php if (!empty($p['fone'])): ?><tr><td>Telefone</td><td><?php echo esc_html($p['fone']); ?></td></tr><?php endif; ?>
          <?php if ($pickup):  ?><tr><td>Ponto de Embarque</td><td><?php echo esc_html($pickup); ?></td></tr><?php endif; ?>
          <?php if ($dropoff): ?><tr><td>Ponto de Desembarque</td><td><?php echo esc_html($dropoff); ?></td></tr><?php endif; ?>

          <tr><td>Categoria/Tarifa</td><td><?php echo esc_html( ger_category_label($cat) ); ?></td></tr>
          <tr><td>Assento</td><td><?php echo $seat ? esc_html($seat) : 'Sem assento'; ?></td></tr>

          <!-- Valores do bilhete -->
          <tr><td>Valor unit.</td><td><?php echo wc_price($valor_unit, ['currency'=>$currency]); ?></td></tr>

          <?php if ($cat !== 'child' || GER_SHOW_TARIFA_ROW_FOR_CHILD): ?>
            <tr><td>Tarifa (unit.)</td><td><?php echo wc_price($tarifa_unit_ticket, ['currency'=>$currency]); ?></td></tr>
          <?php endif; ?>

          <tr><td><strong>Total (valor+tarifa)</strong></td><td><strong><?php echo wc_price($total_bilhete, ['currency'=>$currency]); ?></strong></td></tr>

          <?php if (!empty($extras)): ?>
            <tr><td>Serviços extras</td><td>
              <?php foreach ($extras as $ex): ?>
                <div>
                  <?php
                    echo esc_html($ex['title']);
                    if (!empty($ex['qty']) && $ex['qty'] != 1) echo ' × '.intval($ex['qty']);
                    if (isset($ex['price'])) echo ' — '.wc_price((float)$ex['price'], ['currency'=>$currency]);
                  ?>
                </div>
              <?php endforeach; ?>
            </td></tr>
          <?php endif; ?>
        </table>
      </div>

      <hr class="hr">
      <div class="t-meta">
        <div><strong>Total do bilhete (valor+tarifa)</strong><span><?php echo wc_price($total_bilhete, ['currency'=>$currency]); ?></span></div>
        <div><strong>Desconto do operador (pedido)</strong><span><?php echo wc_price($desconto_operador * -1, ['currency'=>$currency]); ?></span></div>
        <div><strong>Total do pedido</strong><span><?php echo wp_kses_post($order->get_formatted_order_total()); ?></span></div>
      </div>
    </div>

    <?php if ($printed < $total_tickets): ?>
      <hr class="cut">
    <?php endif; ?>

  <?php endfor; endforeach; ?>
  </div>
</div>

<!-- MODAL DE CONFIGURAÇÃO -->
<div class="modal" id="printModal" aria-hidden="true">
  <div class="box">
    <div class="hd">
      <strong>Configurar impressão</strong>
      <button class="btn btn-ghost" id="btn-close-config">Fechar</button>
    </div>
    <div class="bd">
      <div class="tabs">
        <button class="tab active" data-tab="rawbt">RawBT (POS)</button>
        <button class="tab" data-tab="qz">QZ Tray</button>
      </div>

      <!-- Painel RawBT -->
      <div class="panel active" id="panel-rawbt">
        <div class="grid2">
          <div>
            <div class="field"><label>Colunas (CPL)</label><input id="rb-cols" type="number" min="24" max="64" step="1"></div>
            <div class="field"><label>Traço (HR)</label><input id="rb-hr" type="number" min="10" max="64" step="1"></div>
            <div class="field"><label>Linhas extras (final)</label><input id="rb-extra" type="number" min="0" max="30" step="1"></div>
          </div>
          <div>
            <div class="status"><span id="rb-dot" class="dot"></span><span id="rb-status">Aguardando Android/RawBT…</span></div>
            <div class="toolbar">
              <button id="rb-open" class="btn btn-ghost">Abrir RawBT</button>
              <button id="rb-print-all" class="btn btn-green">Imprimir (todos)</button>
              <button id="rb-print-each" class="btn btn-ghost">Imprimir (um a um)</button>
              <button id="rb-save" class="btn btn-blue">Salvar RawBT</button>
            </div>
          </div>
        </div>
      </div>

      <!-- Painel QZ -->
      <div class="panel" id="panel-qz">
        <div class="grid2">
          <div>
            <div class="field"><label>Colunas (CPL)</label><input id="qz-cols" type="number" min="24" max="64" step="1"></div>
            <div class="field"><label>Label min</label><input id="qz-lmin" type="number" min="10" max="40" step="1"></div>
            <div class="field"><label>Label max</label><input id="qz-lmax" type="number" min="10" max="40" step="1"></div>
            <div class="field"><label>Linhas extras (final)</label><input id="qz-extra" type="number" min="0" max="30" step="1"></div>
          </div>
          <div>
            <div class="status"><span id="qz-dot" class="dot"></span><span id="qz-status">QZ Tray: tentando conectar…</span></div>
            <div class="toolbar" style="margin-bottom:8px">
              <button id="qz-connect" class="btn btn-blue">Conectar</button>
              <button id="qz-refresh" class="btn btn-ghost">Atualizar</button>
              <button id="qz-disconnect" class="btn btn-ghost">Desconectar</button>
            </div>
            <div class="toolbar">
              <select id="qz-select" class="qz-select" disabled><option>—</option></select>
              <button id="qz-save" class="btn btn-green" disabled>Salvar impressora</button>
              <button id="qz-save-cfg" class="btn btn-blue">Salvar QZ</button>
            </div>
            <small id="qz-saved" style="display:none;color:#16a34a;margin-top:8px;display:block">Preferências salvas.</small>
          </div>
        </div>
      </div>

    </div>
  </div>
</div>

<!-- JS: impressão + configs + QZ autoconnect (sem QR) -->
<script>
(function(){
  /* ========= ESC/POS base ========= */
  const ESC='\x1B', GS='\x1D';
  const init=ESC+'@', align=n=>ESC+'a'+String.fromCharCode(n);
  const boldOn=ESC+'E'+'\x01', boldOff=ESC+'E'+'\x00';
  const size1x=GS+'!'+'\x00', size2x=GS+'!'+'\x11';
  const feed=n=>ESC+'d'+String.fromCharCode(n);
  const setCP1252 = ESC+'t'+'\x10'; // acentos
  const cutFullFeed = (n=12)=> GS+'V'+'\x42'+String.fromCharCode(Math.max(0,Math.min(255,n)));
  const cutFallback = GS+'V'+'\x00';

  /* ========= Estado & Configs ========= */
  const KEY_RB = 'rawbtCfg_v1';
  const KEY_QZ = 'qzCfg_v1';
  const defaultsRB = { cols: 42, hr: 40, extra: 6 };
  const defaultsQZ = { cols: 48, lmin: 18, lmax: 26, extra: 6, printer: '' };

  function loadCfg(key, defaults){ try{ const v=JSON.parse(localStorage.getItem(key)||''); return {...defaults, ...v}; }catch{ return {...defaults}; } }
  function saveCfg(key, obj){ localStorage.setItem(key, JSON.stringify(obj)); }

  let RB = loadCfg(KEY_RB, defaultsRB);
  let QZCFG = loadCfg(KEY_QZ, defaultsQZ);

  /* ========= Helpers ========= */
  const clean=t=>(t||'').replace(/\s+/g,' ').trim();
  function wrap(str,w){str=String(str??'');if(w<=1)return[str];const words=clean(str).split(/\s+/);const out=[];let line='';for(const wd of words){const tryL=line?(line+' '+wd):wd;if(tryL.length>w){if(line)out.push(line);if(wd.length>w){for(let i=0;i<wd.length;i+=w)out.push(wd.slice(i,i+w));line='';}else{line=wd;}}else{line=tryL;}}if(line)out.push(line);return out;}
  function center(text,w){return wrap(text,w).map(ln=>{const pad=Math.max(0,Math.floor((w-ln.length)/2));return' '.repeat(pad)+ln;}).join('\n')+'\n';}
  function hr(width){return '-'.repeat(width)+'\n';}
  function col2(label,value,width,lw){
    label=clean(label).replace(/:$/,''); value=clean(value);
    const rw=Math.max(1,width-lw);
    const L=wrap(label,lw); const R=wrap(value,rw);
    const rows=Math.max(L.length,R.length); let out='';
    for(let i=0;i<rows;i++){
      const l=L[i]||''; const r=R[i]||''; 
      out+= l + ' '.repeat(Math.max(0,lw-l.length)) + ' '.repeat(Math.max(0,rw-r.length)) + r + '\n';
    }
    return out;
  }
  function scanLabelWidth(ticketEl, width, MIN, MAX){
    const labels=[];
    ticketEl.querySelectorAll('.t-meta div strong').forEach(s=>labels.push(clean(s.textContent)));
    ticketEl.querySelectorAll('.tbl tr').forEach(tr=>{
      const tds=tr.querySelectorAll('td');
      if(tds.length>=2 && !tds[0].hasAttribute('colspan')) labels.push(clean(tds[0].textContent).replace(/:$/,''));
    });
    const maxLen=labels.reduce((m,s)=>Math.max(m,s.length),0);
    const wanted=Math.min(MAX,Math.max(MIN,maxLen+2));
    const minRight=12;
    return Math.min(wanted,Math.max(MIN,width-minRight));
  }
  function encodeCP1252(str){ const bytes=[]; for(let i=0;i<str.length;i++){ const c=str.charCodeAt(i); bytes.push(c<=0xFF?c:0x3F); } return new Uint8Array(bytes); }
  function b64(u8){ let bin=''; for(let i=0;i<u8.length;i++) bin+=String.fromCharCode(u8[i]); return btoa(bin); }

  function ticketNumberLine(idx, total){
    const curr = (idx+1).toString();
    const fmt  = '<?php echo esc_js(GER_TICKET_NUM_FORMAT); ?>';
    return (total>1) ? fmt.replace('%CURR%', curr).replace('%TOTAL%', String(total)) : '';
  }

  /* ========= Builder (RB/QZ) ========= */
  function buildTicketString(ticketEl, cfg, idx, total, brand){
    const cols = cfg.cols;
    const LW   = scanLabelWidth(ticketEl, cols, cfg.lmin ?? 18, cfg.lmax ?? 26);
    const HRL  = Math.min(cols, cfg.hr ?? cols);
    const EXTRA= Math.max(0, cfg.extra ?? 0);

    let out=''; out+=init+setCP1252;
    out+=align(1)+size2x+boldOn+brand+'\n'+boldOff+size1x+align(0);

    const numLine = ticketNumberLine(idx, total);
    if (numLine) out+= numLine + '\n';

    out+=hr(HRL);

    const metas=ticketEl.querySelectorAll('.t-meta')[0];
    if(metas){
      metas.querySelectorAll('div').forEach(d=>{
        const k=d.querySelector('strong')?.textContent||'';
        const v=d.querySelector('span')?.textContent||'';
        if(k||v) out+=col2(k,v,cols,LW);
      });
    }

    const route=ticketEl.querySelector('.route');
    if(route){
      out+=hr(HRL);
      const colsR=route.querySelectorAll('div.col');
      const origem = colsR[0] ? clean(colsR[0].textContent) : '';
      const destino= colsR[1] ? clean(colsR[1].textContent) : '';
      if (origem)  out+=col2('Origem',  origem,  cols, LW);
      if (destino) out+=col2('Destino', destino, cols, LW);
    }

    const tbl=ticketEl.querySelector('.tbl');
    if(tbl){
      out+=hr(HRL);
      tbl.querySelectorAll('tr').forEach(tr=>{
        const tds=tr.querySelectorAll('td'); if(!tds.length) return;
        if(tds.length===1 || tds[0].hasAttribute('colspan')){
          const title=clean(tds[0].textContent); if(title) out+=align(1)+boldOn+title+'\n'+boldOff+align(0);
        } else {
          const k=clean(tds[0].textContent).replace(/:$/,'');
          const v=clean(tds[1].textContent);
          out+=col2(k, v, cols, LW);
        }
      });
    }

    const metaBlocks=ticketEl.querySelectorAll('.t-meta');
    const totals=metaBlocks[metaBlocks.length-1];
    if(totals){
      out+=hr(HRL);
      totals.querySelectorAll('div').forEach(d=>{
        const k=d.querySelector('strong')?.textContent||'';
        const v=d.querySelector('span')?.textContent||'';
        if(k||v) out+=col2(k, v, cols, LW);
      });
    }

    out+= '\n'.repeat(EXTRA);
    out+= feed(6) + cutFullFeed(12) + cutFallback + '\n\n';
    return out;
  }

  function collectTickets(){ const wrap=document.getElementById('ticketWrap'); return wrap ? [...wrap.querySelectorAll('.ticket')] : []; }

  /* ========= Modal & Abas ========= */
  const modal  = document.getElementById('printModal');
  const openBt = document.getElementById('btn-open-config');
  const closeBt= document.getElementById('btn-close-config');
  openBt.addEventListener('click', ()=> modal.classList.add('open'));
  closeBt.addEventListener('click', ()=> modal.classList.remove('open'));
  document.querySelectorAll('.tab').forEach(btn=>{
    btn.addEventListener('click', ()=>{
      document.querySelectorAll('.tab').forEach(b=>b.classList.remove('active'));
      document.querySelectorAll('.panel').forEach(p=>p.classList.remove('active'));
      btn.classList.add('active');
      document.getElementById('panel-'+btn.dataset.tab).classList.add('active');
    });
  });

  function fillInputs(){
    document.getElementById('rb-cols').value  = RB.cols;
    document.getElementById('rb-hr').value    = RB.hr;
    document.getElementById('rb-extra').value = RB.extra;

    document.getElementById('qz-cols').value  = QZCFG.cols;
    document.getElementById('qz-lmin').value  = QZCFG.lmin;
    document.getElementById('qz-lmax').value  = QZCFG.lmax;
    document.getElementById('qz-extra').value = QZCFG.extra;
  }
  fillInputs();

  /* ========= RawBT ========= */
  const dotRB = document.getElementById('dot-rawbt'), txtRB = document.getElementById('txt-rawbt');
  const rbDot= document.getElementById('rb-dot'),   rbStatus=document.getElementById('rb-status');
  const isAndroid = /Android/i.test(navigator.userAgent);
  if(isAndroid){ dotRB.classList.add('ok'); txtRB.textContent='RawBT: pronto'; rbDot.classList.add('ok'); rbStatus.textContent='Android/RawBT detectado'; }
  else { dotRB.classList.add('warn'); txtRB.textContent='RawBT: requer Android'; rbDot.classList.add('warn'); rbStatus.textContent='Use RawBT no Android'; }

  function openRaw(href){ const a=document.createElement('a'); a.href=href; a.style.display='none'; document.body.appendChild(a); a.click(); a.remove(); }

  function rbTicketsString(allParts=true){
    const tickets=collectTickets();
    const brand="<?php echo esc_js($brand); ?>";
    const parts=tickets.map((t,i)=>buildTicketString(t, {cols:RB.cols, hr:RB.hr, extra:RB.extra, lmin:18, lmax:26}, i, tickets.length, brand));
    return allParts ? parts.join('') : parts;
  }

  function printRawBT(all=true){
    const data = all ? rbTicketsString(true) : rbTicketsString(false);
    if(all){
      const b = b64(encodeCP1252(data));
      openRaw('rawbt:base64,'+b);
    }else{
      const parts = rbTicketsString(false);
      (async ()=>{ for(const p of parts){ openRaw('rawbt:base64,'+b64(encodeCP1252(p))); await new Promise(r=>setTimeout(r,800)); }})();
    }
  }

  document.getElementById('btn-print-rawbt').addEventListener('click', ()=>printRawBT(true));
  document.getElementById('rb-open').addEventListener('click', ()=>openRaw('rawbt:'));
  document.getElementById('rb-print-all').addEventListener('click', ()=>printRawBT(true));
  document.getElementById('rb-print-each').addEventListener('click', ()=>printRawBT(false));
  document.getElementById('rb-save').addEventListener('click', ()=>{
    RB.cols  = Math.max(24, Math.min(64, parseInt(document.getElementById('rb-cols').value||RB.cols)));
    RB.hr    = Math.max(10, Math.min(64, parseInt(document.getElementById('rb-hr').value||RB.hr)));
    RB.extra = Math.max(0,  Math.min(30, parseInt(document.getElementById('rb-extra').value||RB.extra)));
    localStorage.setItem('rawbtCfg_v1', JSON.stringify(RB));
    rbStatus.textContent = 'Preferências do RawBT salvas.';
  });

  /* ========= QZ Tray (autoconnect) ========= */
  const dotQZ=document.getElementById('dot-qz'), txtQZ=document.getElementById('txt-qz');
  const CERT_URL = "<?php echo esc_js( get_template_directory_uri() . '/qz/digital-certificate.txt' ); ?>";
  const SIGN_URL = "<?php echo esc_js( get_template_directory_uri() . '/qz/sign-message.php?request=' ); ?>";
  const selQZ = document.getElementById('qz-select');

  function setQZ(msg, cls){
    txtQZ.textContent=msg;
    document.getElementById('qz-status').textContent=msg;
    dotQZ.className='dot '+(cls||'');
    document.getElementById('qz-dot').className='dot '+(cls||'');
  }
  function enableQZButtons(on){
    document.getElementById('btn-print-qz').disabled = !on;
    document.getElementById('qz-save').disabled = !on;
  }

  async function ensureQZ(){
    if (typeof qz==='undefined' || !qz.websocket) throw new Error('qz-tray.js não encontrado.');
    if (qz.websocket.isActive()) return;
    qz.security.setCertificatePromise((resolve,reject)=>{
      fetch(CERT_URL,{cache:'no-store'}).then(r=>r.ok?r.text():Promise.reject('Certificado ausente')).then(resolve).catch(reject);
    });
    qz.security.setSignatureAlgorithm("SHA512");
    qz.security.setSignaturePromise(toSign=>(resolve,reject)=>{
      fetch(SIGN_URL+encodeURIComponent(toSign),{cache:'no-store'}).then(r=>r.ok?r.text():Promise.reject('Assinatura falhou')).then(resolve).catch(reject);
    });
    await qz.websocket.connect();
  }

  async function listPrinters(){
    let names=[];
    if (qz.printers?.find)       names = await qz.printers.find();
    else if (qz.printers?.details){ const det=await qz.printers.details(); names=Array.isArray(det)?det.map(p=>p?.name||p):[]; }
    selQZ.innerHTML='';
    if(!names.length){
      selQZ.appendChild(new Option('Nenhuma encontrada',''));
      selQZ.disabled=true; enableQZButtons(false);
      setQZ('Conectado ao QZ, mas sem impressoras','warn'); return false;
    }
    names.forEach(n=> selQZ.appendChild(new Option(n,n)));
    selQZ.disabled=false; enableQZButtons(true);
    if(QZCFG.printer && names.includes(QZCFG.printer)){ selQZ.value = QZCFG.printer; setQZ('QZ pronto: '+QZCFG.printer,'ok'); }
    else { setQZ('QZ conectado. Selecione e salve a impressora.','ok'); }
    return true;
  }

  async function autoConnectQZ(){
    try{
      await ensureQZ();
      const ok = await listPrinters();
      if(ok && QZCFG.printer){
        document.getElementById('btn-print-qz').disabled=false;
        setQZ('QZ pronto: '+QZCFG.printer,'ok');
      }
    }catch(e){
      setQZ('QZ Tray não está em execução. Abra o app.','err');
      document.getElementById('btn-print-qz').disabled=true;
    }
  }
  document.addEventListener('DOMContentLoaded', autoConnectQZ);

  document.getElementById('qz-connect').addEventListener('click', autoConnectQZ);
  document.getElementById('qz-refresh').addEventListener('click', async ()=>{
    try{ await ensureQZ(); await listPrinters(); }catch(e){ setQZ('Erro: '+(e?.message||e),'err'); }
  });
  document.getElementById('qz-disconnect').addEventListener('click', async ()=>{
    try{ if(qz?.websocket?.isActive()) await qz.websocket.disconnect(); setQZ('QZ desconectado.',''); enableQZButtons(false); }catch(e){ setQZ('Erro: '+(e?.message||e),'err'); }
  });

  document.getElementById('qz-save').addEventListener('click', ()=>{
    const val=selQZ.value||'';
    if(!val){ alert('Escolha uma impressora'); return; }
    QZCFG.printer = val;
    localStorage.setItem('qzCfg_v1', JSON.stringify(QZCFG));
    setQZ('Impressora salva: '+val, 'ok');
    document.getElementById('btn-print-qz').disabled=false;
  });

  document.getElementById('qz-save-cfg').addEventListener('click', ()=>{
    QZCFG.cols  = Math.max(24, Math.min(64, parseInt(document.getElementById('qz-cols').value||QZCFG.cols)));
    QZCFG.lmin  = Math.max(10, Math.min(40, parseInt(document.getElementById('qz-lmin').value||QZCFG.lmin)));
    QZCFG.lmax  = Math.max(10, Math.min(40, parseInt(document.getElementById('qz-lmax').value||QZCFG.lmax)));
    QZCFG.extra = Math.max(0,  Math.min(30, parseInt(document.getElementById('qz-extra').value||QZCFG.extra)));
    localStorage.setItem('qzCfg_v1', JSON.stringify(QZCFG));
    setQZ('Preferências do QZ salvas.', 'ok');
  });

  async function printQZ(){
    if(!QZCFG.printer){ document.getElementById('printModal').classList.add('open'); setQZ('Escolha/salve uma impressora no QZ.','warn'); return; }
    try{
      await ensureQZ();
      const cfg=qz.configs.create(QZCFG.printer,{encoding:'CP1252'});
      const brand="<?php echo esc_js($brand); ?>";
      const tickets=collectTickets();
      const parts=tickets.map((t,i)=>buildTicketString(t, {cols:QZCFG.cols, hr:QZCFG.cols, extra:QZCFG.extra, lmin:QZCFG.lmin, lmax:QZCFG.lmax}, i, tickets.length, brand));
      await qz.print(cfg, [{type:'raw', format:'plain', data: parts.join('')}]);
      setQZ('Impresso em: '+QZCFG.printer, 'ok');
    }catch(e){
      setQZ('Erro ao imprimir: '+(e?.message||e),'err');
    }
  }
  document.getElementById('btn-print-qz').addEventListener('click', printQZ);
})();
</script>

<!-- QZ Tray (mantenha este arquivo no tema) -->
<script src="<?php echo get_template_directory_uri(); ?>/qz/js/qz-tray.js"></script>

<?php get_footer(); ?>
